import logging
import ujson as json
from datetime import datetime

from dateutil.tz import tzlocal

from mail.python.theatre.logging.request_id import REQUEST_ID


def monkey_patch_logger():
    def makeRecord(self, name, level, fn, lno, msg, args, exc_info,
                   func=None, extra=None, sinfo=None):
        rv = self.__makeRecord(
            name, level, fn, lno, msg, args, exc_info, func=func, extra=extra, sinfo=sinfo
        )
        extra = extra or {}
        if 'request_id' not in extra and REQUEST_ID.get():
            extra['request_id'] = REQUEST_ID.get()
        rv.__dict__['extra'] = extra
        return rv

    logging.getLoggerClass().__makeRecord = logging.getLoggerClass().makeRecord
    logging.getLoggerClass().makeRecord = makeRecord


class ExtraJsonFormatter(logging.Formatter):
    """
    Format message as json entry, putting some additional predefined attributes.
    Also add anything that was provided as `extra` keyword parameter in `Logger.log()` call,
    if `monkey_patch_logger()` was called.
    """
    TZ = tzlocal()

    def format(self, record: logging.LogRecord):
        message = super(ExtraJsonFormatter, self).format(record)
        ts = datetime.fromtimestamp(record.created, tz=self.TZ)
        entry = {
            'log_level': record.levelname,
            'logger_name': record.name,
            'datetime': ts.isoformat(),
            'unixtime_ms': int(ts.timestamp() * 1000),
            'message': message,
        }
        if hasattr(record, 'extra'):
            entry.update(record.extra)
        return json.dumps(entry)
