import ssl
from _ssl import PROTOCOL_TLSv1_2
from typing import Mapping, Dict

import asyncpg
from asyncpg.pool import Pool
from sqlalchemy.engine.url import URL

from mail.python.theatre.app.log_helpers.postgre_conn import LoggingConn
from .base import SettingsNode


class DbSettings(SettingsNode):
    name: str = None
    user: str = None
    password: str = None
    host: str = None
    host_ro: str = None
    port: int = None
    min_size: int = 1
    max_size: int = 4
    timeout_sec: int = 3
    command_timeout_sec: int = 10

    ssl = ssl.SSLContext(protocol=PROTOCOL_TLSv1_2)

    def pg_dsn(self, ro: bool = False, **kwargs) -> str:
        url_kwargs = dict(
            database=self.name,
            password=self.password,
            host=self.host if not ro else self.host_ro,
            port=self.port,
            username=self.user,
            drivername='postgres',
        )
        url_kwargs.update(kwargs)
        return str(URL(**url_kwargs))

    async def create_pool(self, ro: bool = False, **kwargs) -> Pool:
        pool_kwargs = dict(
            min_size=self.min_size,
            max_size=self.max_size,
            ssl=self.ssl,
            timeout=self.timeout_sec,
            command_timeout=self.command_timeout_sec,
            connection_class=LoggingConn,
        )
        pool_kwargs.update(kwargs)
        return await asyncpg.create_pool(
            dsn=self.pg_dsn(ro=ro),
            **pool_kwargs
        )


class MultihostDbSettings(DbSettings):
    hosts: Dict[str, str] = {}
    health_check_limit: int = 10

    def dsn_for_host(self, host: str):
        return str(URL(
            database=self.name,
            password=self.password,
            host=host,
            port=self.port,
            username=self.user,
            drivername='postgres',
        ))

    async def create_pools(self) -> Mapping[str, Pool]:
        hosts = self.hosts or {'default': self.host}

        async def pool_for_host(host: str):
            return await asyncpg.create_pool(
                dsn=self.dsn_for_host(host),
                min_size=self.min_size,
                max_size=self.max_size,
                ssl=self.ssl,
                timeout=self.timeout_sec,
                command_timeout=self.command_timeout_sec,
                connection_class=LoggingConn,
            )

        return {dc: await pool_for_host(host) for dc, host in hosts.items()}
