import click
import os
import yaml
import asyncio

from typing import Mapping, Union
from datetime import timedelta
from pathlib import Path
from aiohttp import web
from yaml import SafeLoader, MappingNode, ScalarNode

from mail.python.theatre.app.settings import LogSettings
from mail.python.theatre.app.log_helpers.app_logs import init_app_logging

from mail.python.theatre.common_applications.pg_dbstats.lib.dbstat import make_db_stat_director


def init_logging(config: Mapping[str, Union[str, Mapping]]) -> None:
    log_config = config['log']

    log = LogSettings()
    log.dir_path = Path(log_config['dir'])
    log.level = log_config['level']
    init_app_logging(settings=log, env=None)


async def create_app(config: Mapping[str, Union[str, Mapping]]) -> web.Application:
    import uvloop
    asyncio.set_event_loop_policy(uvloop.EventLoopPolicy())

    app = web.Application(loop=asyncio.get_event_loop())

    dbstat = make_db_stat_director(config)

    async def shutdown(_):
        await dbstat.stop()

    await dbstat.start()

    app.router.add_get('/solomon', dbstat.solomon_handler, name='solomon')
    app.on_shutdown.append(shutdown)

    return app


@click.command()
@click.option('--host', default='::0', help='IP addr to bind to')
@click.option('--port', default=8080, help='HTTP port to listen')
@click.option('--dir', 'working_dir', default='.', help='Working directory')
@click.option('--config', 'config_file', required=True, help='Configuration file')
def main(host, port, working_dir, config_file):
    os.chdir(working_dir)

    def env_constructor(loader: SafeLoader, node: ScalarNode) -> str:
        return os.environ.get(loader.construct_scalar(node))

    def timedelta_yaml_constructor(loader: SafeLoader, node: MappingNode) -> timedelta:
        return timedelta(**loader.construct_mapping(node))

    yaml.add_constructor('!Env', env_constructor, Loader=yaml.SafeLoader)
    yaml.add_constructor('!TimeDelta', timedelta_yaml_constructor, Loader=yaml.SafeLoader)

    with open(config_file, 'r') as f:
        config = yaml.safe_load(f)

    init_logging(config)

    web.run_app(
        create_app(config),
        port=port,
        host=host,
        access_log_format='%a %t "%r" %s %b "%{Referer}i" "%{User-Agent}i" %Tf'
    )


if __name__ == '__main__':
    main()
