from typing import Type

from library.python.monlib.metric_registry import MetricRegistry

import warnings
from mail.python.theatre.stages.db_stats.roles.db_stat_poller import DbStatPoller


def solomon_registry(poller_cls: Type[DbStatPoller]):
    """
    После каждого сбора метрик складывает новые значения в MetricRegistry - внутреннее представление метрик из
    клиентской библиотеки Соломона.
    Использование этого представления позволяет переиспользовать код сериализации, в том числе в формате SPACK.
    Для всех метрик используется тип GAUGE (текущее значение некоего дробного числа), как наиболее отражающий
    физический смысл значения, получаемого регулярным выполнением запроса в БД:
    * IGAUGE и COUNTER не подходят, т.к. в результате запроса может быть не целочисленная величина
    * RATE не подходит, т.к. dbstats не умеет делать инкрементные запросы, зависящие от результата предыдущего выполнения
    * HIST не подходит, т.к. dbstats не рассчитан на запросы с большим количеством возвращаемых значений, которые имело
        бы смысл свернуть в гистограмму одного сигнала

    N.B. Соломон разрешает использовать максимум 16 меток, из них:
        * 3 зарезервированы самим Соломоном (project, cluster, service)
        * 1 добавляется Соломон-агентом (host).
        * 2 добавляется dbstats-ом (dbname, name и value).
        Итого остается 9 меток.
        Наличие в запросе большего количества меток приведет к потере данных.
    """
    class SolomonRegistryPoller(poller_cls):
        def dump_solomon_metrics(self, solomon_registry: MetricRegistry):
            for db_host, host_signals in self._signals.items():
                for signal in host_signals:
                    if len(signal._tags) > 9:
                        warnings.warn(f'Signal {signal.name} labels count is {len(signal._tags)} but allowed maximum is 10')
                    solomon_registry.gauge(
                        dict(
                            dbname=self._db.prj,
                            name=signal.name,
                            **{k: str(v) for k, v in signal._tags.items() if v is not None}
                        )
                    ).set(
                        signal.value
                    )

    return SolomonRegistryPoller
