from array import array
from bisect import bisect_right
import logging
from typing import Sequence, Iterable

from .typing import HistBucket, HistMetric, Metrics

log = logging.getLogger(__name__)


class JobProfile:
    def __init__(self, name: str, waitable: bool = False):
        self._name = name
        self._success_hist_msec = Hist(buckets=make_buckets(0.1, 10, 500, 3000), name=f'{name}_success_ms')
        self._fail_hist_msec = Hist(buckets=make_buckets(0.1, 10, 500, 3000), name=f'{name}_fail_ms')
        self._wait_hist_msec = waitable and Hist(
            buckets=make_buckets(10, 1000, 5000, 30000), name=f'{name}_wait_ms'
        )

    def ok(self, sec: float):
        self._success_hist_msec.update(sec * 1000)

    def failed(self, sec: float):
        self._fail_hist_msec.update(sec * 1000)

    def waited(self, sec: float):
        if not self._wait_hist_msec:
            log.warning(f'{self._name} job profile is misconfigured, trying to profile wait time but it is disabled')
        else:
            self._wait_hist_msec.update(sec * 1000)

    def get(self) -> Metrics:
        return [m.get() for m in [self._success_hist_msec, self._fail_hist_msec, self._wait_hist_msec] if m]


class Hist:
    def __init__(self, buckets: Sequence[float], name: str, suffix: str = 'hgram'):
        self._buckets = buckets
        self._name = name
        self._counters = array('Q', [0] * len(buckets))
        self._suffix = suffix

    def update(self, value: float):
        index = bisect_right(self._buckets, value)
        if index <= 0:
            return
        self._counters[index - 1] += 1

    def get(self) -> HistMetric:
        return f'{self._name}_{self._suffix}', sorted((k, v) for k, v in zip(self._buckets, self._counters))


def make_buckets(left, mid, right, timeout):
    """
    Makes continuous integer series from given reference values with len(result) == 50
    :return: [0, ..., left, ..., mid, ..., right, ..., timeout, ..., 2*timeout]
    """
    def stepped_range(_left, _right, count):
        assert _right > _left
        step = (_right - _left) / float(count)
        return [_left + step * v for v in range(count)]

    return (
        stepped_range(0, left, 5)
        + stepped_range(left, mid, 20)
        + stepped_range(mid, right, 15)
        + stepped_range(right, timeout, 5)
        + [(1 + 0.2 * i) * timeout for i in range(5)]
    )


def make_hist_from_counter(name: str, counter: Iterable[HistBucket]) -> HistMetric:
    return f'{name}_hgram', sorted((k, v) for k, v in counter)
