import asyncio
import logging
from typing import Coroutine, List, Callable, Dict, Any

from mail.python.theatre.profiling.typing import Metrics

log = logging.getLogger(__name__)


class Attendant:
    """The very base role, can be started and stopped"""
    def __init__(self, role_name: str = None, **_):
        self._stopped = False
        self._name = role_name

    async def start(self):
        self._stopped = False
        log.info('%s is started', self.name)

    async def stop(self, wait=False):  # noqa
        self._stopped = True
        log.info('%s is stopped', self.name)

    @property
    def name(self) -> str:
        # Getattr because this could be called before Attendant.__init__
        return getattr(self, '_name', None) or f'{self.__class__.__name__}'

    def metrics(self) -> Metrics:
        return []

    def state(self) -> Dict[str, Any]:
        return {}


class Servant(Attendant):
    """Role that has endless routine to be worked on. Routine can be multiplied"""
    def __init__(self, routine: Callable[[], Coroutine], servant_count=1, **kwargs):
        super().__init__(**kwargs)
        self._routine = routine
        self._servant_count = servant_count
        self._tasks: List[asyncio.Task] = []

    async def start(self):
        self._tasks = [
            asyncio.get_event_loop().create_task(self._routine())
            for _ in range(self._servant_count)
        ]
        await super().start()

    async def stop(self, wait=False):
        await super().stop(wait=wait)
        if not wait:
            [t.cancel() for t in self._tasks]
        try:
            await asyncio.wait(self._tasks)
        except asyncio.CancelledError:
            pass
