import asyncio
import logging
import random
from datetime import timedelta
from time import monotonic

from mail.python.theatre.logging.request_id import set_request_id
from mail.python.theatre.profiling.hist import JobProfile
from mail.python.theatre.profiling.typing import Metrics
from mail.python.theatre.typing import ArglessJob
from mail.python.theatre.utils.nested_job import nested_job
from .base import Servant

log = logging.getLogger(__name__)


class Cron(Servant):
    """Role that runs given job every `run_every` time period"""
    def __init__(self, job: ArglessJob, run_every: timedelta, randomize_start_time: bool = False, **kwargs):
        super().__init__(routine=nested_job(self._cron_routine, job), **kwargs)
        self._run_every_sec = run_every.total_seconds()
        self._job_profile = JobProfile(name=self.name)
        self._randomize_start_time = randomize_start_time

    async def _cron_routine(self, job: ArglessJob):
        while not self._stopped:
            start = monotonic()

            if self._randomize_start_time:
                sleep_before = random.random() * self._run_every_sec
                await asyncio.sleep(sleep_before)

            with set_request_id(prefix=self.name):
                try:
                    await job()
                    elapsed_sec = monotonic() - start
                    self._job_profile.ok(elapsed_sec)
                except Exception as e:
                    elapsed_sec = monotonic() - start
                    self._job_profile.failed(elapsed_sec)
                    log.exception(e)

            sleep_for = max(.0, self._run_every_sec - elapsed_sec)
            await asyncio.sleep(sleep_for)

    def metrics(self) -> Metrics:
        return self._job_profile.get()
