import logging
from asyncio import QueueFull, Queue
from datetime import datetime, timedelta
from typing import Callable, Any, Coroutine

import dateutil.tz

from mail.python.theatre.utils.nested_job import nested_job
from .delayed import Delayed
from .director import Director
from .queue import Queued

log = logging.getLogger(__name__)


class DelayItem(Exception):
    def __init__(self, delay_until: datetime):
        self.delay_until = delay_until


class DelayedQueued(Director):
    """
    Composite role that schedules delayed event processing via Delayed role, and process ready events via Queue role.
    """
    TZ = dateutil.tz.tzlocal()

    def __init__(
            self,
            job: Callable[[Any], Coroutine],
            # Delayed args
            max_delayed: int,
            retry_delayed: timedelta,
            # Queued args
            servant_count: int = 1,
            queue: Queue = None,
            **kwargs
    ):
        self._delayed = Delayed(
            consumer=self._on_wait, max_delayed=max_delayed, retry_delayed=retry_delayed,
            role_name=f'{self.name}_delayed',
        )
        self._queued = Queued(
            job=nested_job(self._dq_job, job), servant_count=servant_count, queue=queue,
            role_name=f'{self.name}_queued',
        )
        super().__init__(tasks=[self._queued, self._delayed])

    def put(self, item, now: datetime = None):
        if now is None:
            now = datetime.now(tz=self.TZ)
        if item.run_at > now:
            self._delayed.put_delayed(item, at=item.run_at)
        else:
            self._queued.put_nowait(item)

    def _on_wait(self, item) -> bool:
        try:
            self._queued.put_nowait(item)
            return True
        except QueueFull:
            log.warning("%s queue is full, can't process", self._queued.name)
            return False

    async def _dq_job(self, job, item):
        try:
            return await job(item)
        except DelayItem as e:
            self._delayed.put_delayed(item, e.delay_until)
