import asyncio
import logging
from time import monotonic
from typing import Coroutine, Callable

from mail.python.theatre.profiling.hist import JobProfile
from mail.python.theatre.profiling.typing import Metrics
from mail.python.theatre.utils.nested_job import nested_job
from .base import Servant

log = logging.getLogger(__name__)


class Detached(Servant):
    """
    Role that encapsulate some arg-less job and synchronize all requests to run that job via `asyncio.Event`. Job launch
        is fire'n'forget from client side. This is useful for jobs that must not run concurrently with itself.
    Does not guarantee to run given job same count of times that `run_once` was called. Any `run_once` calls are ignored
        if job run is already planned. Please consider using `roles.Queued` if you need fair queueing of run requests.
    """
    def __init__(self, job: Callable[[], Coroutine], **kwargs):
        self._event = asyncio.Event()
        super().__init__(routine=nested_job(self._detached_routine, job), **kwargs)
        self._job_profile = JobProfile(name=self.name, waitable=True)

    def run_once(self):
        self._event.set()

    async def stop(self):
        self._stopped = True
        self.run_once()
        await super().stop()

    async def _detached_routine(self, job):
        while not self._stopped:
            start = monotonic()
            await self._event.wait()
            waited = monotonic()
            self._job_profile.waited(monotonic() - start)
            self._event.clear()
            try:
                await job()
                self._job_profile.ok(monotonic() - waited)
            except BaseException as e:
                log.error('%s failed routine run', self.name)
                log.exception(e)
                self._job_profile.failed(monotonic() - waited)
            # Event.wait() won't reschedule if event is set, so we force reschedule here
            await asyncio.sleep(0)

    def metrics(self) -> Metrics:
        return self._job_profile.get()
