import asyncio
from itertools import chain
from typing import Sequence, Dict

from mail.python.theatre.profiling.typing import Metrics
from .base import Attendant


class Director(Attendant):
    """Role that is basically implements `Composite` pattern. Aggregates list of other roles."""
    def __init__(self, tasks: Sequence[Attendant], **kwargs):
        super().__init__(**kwargs)
        self.__tasks = tasks

    async def start(self):
        for t in self.__tasks:
            await t.start()
        await super().start()

    async def stop(self, wait=False):
        # Asyncio calls this twice sometimes, for unknown reason
        if self._stopped:
            return
        await super().stop(wait=wait)
        # TODO :: better diagnostics for failed graceful shutdown
        if self.__tasks:
            done, pending = await asyncio.wait([t.stop() for t in self.__tasks], timeout=10)
            for task in pending:
                print(task)

    def tasks(self) -> Dict[str, Attendant]:
        assert not self._stopped
        return {t.name: t for t in self.__tasks}

    def metrics(self) -> Metrics:
        return list(chain.from_iterable(task.metrics() for task in self.__tasks))
