import asyncio
import logging
from asyncio import Queue
from time import monotonic
from typing import Callable, Any, Coroutine

from mail.python.theatre.profiling.hist import JobProfile
from mail.python.theatre.profiling.typing import Metrics
from mail.python.theatre.utils.nested_job import nested_job
from .base import Servant

log = logging.getLogger(__name__)


class PoisonPill:
    pass


class Queued(Servant):
    """Role that encapsulate given job and run it against every item given passed into `add` method."""
    def __init__(self, job: Callable[[Any], Coroutine], queue: Queue = None, **kwargs):
        super().__init__(
            routine=nested_job(self._queued_routine, job),
            **kwargs
        )
        self._queue = queue or Queue()
        self._job_profile = JobProfile(name=self.name, waitable=True)

    async def stop(self):
        for _ in range(self._servant_count):
            await self._queue.put(PoisonPill)
        await super().stop()
        # TODO :: Doesnt' work don't know why
        # await self._queue.join()

    async def _queued_routine(self, job):
        while not self._stopped or self._queue.qsize():
            start = monotonic()
            item = await self._queue.get()
            if item is PoisonPill:
                self._queue.task_done()
                return
            waited = monotonic()
            self._job_profile.waited(waited - start)
            try:
                await job(item)
                self._job_profile.ok(monotonic() - waited)
            except Exception as e:
                self._job_profile.failed(monotonic() - waited)
                log.error('%s failed item processing', self.name)
                log.exception(e)
            self._queue.task_done()
            # Queue.get() won't reschedule if queue is not empty, so we force reschedule here
            await asyncio.sleep(0)

    def put_nowait(self, item):
        self._queue.put_nowait(item)

    def metrics(self) -> Metrics:
        return self._job_profile.get() + [
            (f'{self._name}_qsize_ammx', self._queue.qsize())
        ]
