import asyncio
import datetime
from time import time
from typing import Callable, Coroutine

from mail.python.theatre.utils.derived_job import derived_class_in_qualname
from .detach import Detached


class Timer(Detached):
    """
    Role that plans delayed execution of given job. If job is already planned, merges any further requests to run,
        executing given job once at earliest timestamp given. Thus, timestamp is updated any time `run_at` is called,
        if new timestamp is earlier that previous.
    """
    MAX_ASYNCIO_SLEEP_SECS = 60 * 60 * 24

    def __init__(self, job: Callable[[], Coroutine], **kwargs):
        self._wakeup_at: float = None
        self._wakeup_task = None
        super().__init__(job=job, **kwargs)

    async def stop(self):
        if self._wakeup_task:
            self._wakeup_task.cancel()
        await super().stop()

    def run_at(self, at: datetime.datetime):
        return self.run_at_unixtime(at.timestamp())

    def run_at_unixtime(self, at: float):
        if self._wakeup_task:
            if self._wakeup_at and self._wakeup_at <= at:
                return
            self._wakeup_task.cancel()
        self._wakeup_at = at
        self._wakeup_task = asyncio.get_event_loop().create_task(
            derived_class_in_qualname(self._sleep_and_run)
        )

    async def _sleep_and_run(self):
        sleep_for = self._wakeup_at - time()
        while sleep_for >= self.MAX_ASYNCIO_SLEEP_SECS:
            await asyncio.sleep(self.MAX_ASYNCIO_SLEEP_SECS)
            sleep_for -= self.MAX_ASYNCIO_SLEEP_SECS
        if sleep_for > 0:
            await asyncio.sleep(sleep_for)
        self.run_once()
        self._wakeup_task = None

    def state(self):
        return {
            'wakeup_at': datetime.datetime.utcfromtimestamp(self._wakeup_at)
        }
