import asyncio
from abc import abstractmethod
from typing import Type

import aiomonitor
from aiohttp import web
from terminaltables import AsciiTable

from mail.python.theatre.app.env import EnvType, qloud_env
from mail.python.theatre.app.log_helpers.app_logs import init_app_logging
from mail.python.theatre.app.settings import SettingsNode
from mail.python.theatre.roles import Director


class Stage:
    def __init__(self, env: str, name: str, settings: Type[SettingsNode] = SettingsNode):
        self.app = web.Application()
        self.env = env and EnvType(env) or qloud_env()
        assert self.env
        self.settings = settings.from_env(self.env)
        self.app.update(name=name, settings=self.settings)

        init_app_logging(settings=self.settings.log, env=self.env)

    def run(self, host: str = '::1', port: int = 8080):
        with aiomonitor.start_monitor(
                monitor=StageMonitor.with_stage(self),
                loop=asyncio.get_event_loop(),
                host='localhost',
                port=port + 1,
                console_port=port + 2,
                locals={'stage': self, **vars(self)}
        ):
            web.run_app(
                self.create_app(), port=port, host=host,
                access_log_format='%a %t "%r" %s %b "%{Referer}i" "%{User-Agent}i" "%{X-Request-Id}o" %Tf'
            )

    def roles(self):
        director: Director = self.app['director']
        return director.tasks()

    @abstractmethod
    async def create_app(self) -> web.Application:
        pass


class StageMonitor(aiomonitor.Monitor):
    def __init__(self, stage: Stage, *args, **kwargs):
        self.stage = stage
        super(StageMonitor, self).__init__(*args, **kwargs)

    @staticmethod
    def with_stage(stage: Stage):
        class MonitorImpl(StageMonitor):
            def __init__(self, *args, **kwargs):
                super(MonitorImpl, self).__init__(stage, *args, **kwargs)
        return MonitorImpl

    def do_roles(self):
        table = AsciiTable([('Roles',)] + [(name,) for name, role in self.stage.roles().items()])
        self._sout.write(table.table)
        self._sout.write('\n')
        self._sout.flush()
