-- name: count_workers
SELECT count(*)
  FROM buckets.workers
 WHERE worker != :worker
   AND now() - heartbeated < make_interval(secs => :heartbeat_deadline_secs)

-- name: count_buckets
SELECT count(*)
  FROM buckets.buckets

-- name: acquire_buckets
WITH ready_buckets AS (
  SELECT bucket_id
    FROM buckets.buckets
         -- Select buckets that has been already taken by this worker
   WHERE worker = :worker
         -- And buckets that have outdated heartbeat
      OR heartbeat IS NULL
      OR now() - heartbeat > make_interval(secs => :heartbeat_deadline_secs)
   ORDER BY
         -- Take buckets already in process by this worker first,
         bucket_id = any(:bucket_ids::integer[]) DESC,
         -- Then take buckets that was taken by this worker previously
         worker IS NOT DISTINCT FROM :worker DESC,
         -- Then take any other buckets
         bucket_id
   LIMIT :bucket_limit
  FOR NO KEY UPDATE SKIP LOCKED
), updated_buckets as (
    UPDATE buckets.buckets b
       SET worker = :worker,
           heartbeat = now()
      FROM ready_buckets rb
     WHERE b.bucket_id = rb.bucket_id
    RETURNING b.bucket_id
), released_buckets as (
    UPDATE buckets.buckets
       SET heartbeat = NULL
     WHERE worker = :worker
       AND bucket_id NOT IN (SELECT * FROM updated_buckets)
) SELECT * FROM updated_buckets

-- name: release_buckets
UPDATE buckets.buckets
   SET heartbeat = now() - make_interval(secs => :heartbeat_deadline_secs - :restart_deadline_secs)
 WHERE worker = :worker

-- name: register_worker
INSERT INTO buckets.workers (worker, heartbeated)
VALUES (:worker, now())
ON CONFLICT (worker) DO UPDATE SET heartbeated = now()

-- name: retire_worker
DELETE FROM buckets.workers
 WHERE worker = :worker
