import logging
from datetime import datetime, timedelta
from typing import List

from psycopg2 import Error as PGError

from mail.python.theatre.roles import Cron
from mail.python.theatre.stages.bucket_holder.interactions.db.buckets import BucketsQueries
from mail.python.theatre.stages.bucket_holder.typing import BucketId

from ..settings.bucket_holder import BucketHolderSettings

log = logging.getLogger(__name__)


class BucketHolder(Cron):
    """Acquires buckets in DB, using heartbeat-based ownership scheme"""

    def __init__(self, queries: BucketsQueries, settings: BucketHolderSettings):
        self._settings = settings
        self._queries = queries
        super().__init__(job=self.acquire, **settings.cron.as_dict())
        # TODO :: Subject to finegrain setup
        self._deadline_secs = self._run_every_sec * 4
        self._restart_time_secs = self._run_every_sec
        # State
        self._target_bucket_cnt: int = self._settings.bucket_limit
        self.ack_bucket_ids: List[BucketId] = []
        self.proc_bucket_ids: List[BucketId] = []
        self._ack_ts = datetime.now()

    @property
    def deadline_secs(self):
        return self._deadline_secs

    @property
    def target_bucket_cnt(self) -> int:
        return self._target_bucket_cnt

    @target_bucket_cnt.setter
    def target_bucket_cnt(self, value: int):
        self._target_bucket_cnt = min(value, self._settings.bucket_limit)

    async def stop(self, *args, **kwargs):
        await super().stop(*args, **kwargs)
        await self._queries.release_buckets(
            deadline_secs=self._deadline_secs,
            time_to_restart_secs=self._restart_time_secs
        )
        self.ack_bucket_ids = []

    async def acquire(self):
        try:
            self.ack_bucket_ids = await self._queries.acquire_buckets(
                bucket_ids=self.proc_bucket_ids,
                limit=self.target_bucket_cnt,
                deadline_secs=self.deadline_secs
            )
            self._ack_ts = datetime.now()
        except PGError as e:
            log.exception(e)
            if self._ack_ts < datetime.now() - timedelta(seconds=self.deadline_secs):
                self.ack_bucket_ids = []
