import asyncio
from itertools import chain
from typing import Callable, Dict, Coroutine

from mail.python.theatre.roles import Attendant
from mail.python.theatre.profiling.typing import Metrics

from mail.python.theatre.stages.bucket_holder.typing import BucketId

ProcessorFactoryT = Callable[[int], Coroutine]


class BucketsProcessor(Attendant):
    """Maintains set of running processors, one per bucket"""
    def __init__(self, processor_factory: ProcessorFactoryT):
        self.processors: Dict[BucketId, Attendant] = {}
        self._factory = processor_factory
        super().__init__()

    async def attach(self, bucket_id: BucketId):
        if bucket_id in self.processors.keys():
            pass
        self.processors[bucket_id] = processor = await self._factory(bucket_id)
        await processor.start()

    async def detach(self, bucket_id: BucketId, wait=True):
        processor = self.processors.pop(bucket_id, None)
        if processor is None:
            pass
        await processor.stop(wait=wait)

    async def stop(self, wait=False):
        # Asyncio calls this twice sometimes, for unknown reason
        if self._stopped:
            return
        await super().stop(wait=wait)
        # TODO :: better diagnostics for failed graceful shutdown
        if self.processors:
            done, pending = await asyncio.wait(
                [p.stop() for p in self.processors.values()],
                timeout=10
            )
            for task in pending:
                print(task)

    def metrics(self) -> Metrics:
        return list(chain.from_iterable(task.metrics() for task in self.processors.values()))
