import asyncio
from time import time

from mail.python.theatre.utils.coro_interceptor import InterceptorLogic


class CoroThrottle(InterceptorLogic):
    """
    Throttles coroutine execution intercepted by `CoroInterceptor` by sleep before every reenter.
    Sleeps for a `self.throttle_share` fraction multiplied by time passed after previous (re)enter
    """
    def __init__(self, min_throttle=0.0, max_throttle=0.9, *args, **kwargs):
        self.throttle_share = 0.0
        self.min_throttle = min_throttle
        self.max_throttle = max_throttle
        self.start = time()

        super().__init__(*args, **kwargs)

    def inc(self, delta: float = 0.05):
        self.throttle_share = min(self.max_throttle, self.throttle_share + delta)

    def dec(self, delta: float = 0.05):
        self.throttle_share = max(self.min_throttle, self.throttle_share - delta)

    def time_to_throttle(self) -> float:
        return (time() - self.start) * self.throttle_share / (1 - self.throttle_share)

    async def on_reenter(self):
        assert self.start
        await asyncio.sleep(self.time_to_throttle())
        self.start = time()
