from functools import wraps
from typing import Callable, Any, Coroutine


def derived_class_in_qualname(bound_method: Callable[[Any], Coroutine]):
    """
    For a base class async method to be called with derived class object,
    patches it's string representation with derived class name.
    This provides meaningful string representation of asyncio.Task objects created from such methods.

    Example:

    In [1]: class Base:
       ...:     async def foo(self):
       ...:         pass
       ...:

    In [2]: class Derived(Base):
       ...:     pass
       ...:

    In [3]: Derived().foo()
    Out[3]: <coroutine object Base.foo at 0x108a6e1c8>

    In [4]: coro = derived_class_in_qualname(Derived().foo)
    Out[4]: <coroutine object Derived :: Base.foo at 0x1088b2dc8>
    """
    @wraps(bound_method)
    async def impl(*args, **kwargs):
        return await bound_method(*args, **kwargs)

    cls = type(bound_method.__self__)
    impl.__qualname__ = f'{cls.__qualname__} :: {bound_method.__qualname__}'
    return impl()
