import inspect
from functools import wraps
from typing import TypeVar, Callable, Any

NestedJob = TypeVar('NestedJob', bound=Callable)


def nested_job(method: Callable[[NestedJob], Any], nested: NestedJob) -> Callable[[NestedJob], Any]:
    """
    Given a function that wraps some nested job, patches wrapper's string representation so it contains nested job's name.
    This provides meaningful string representation of asyncio.Task objects created from such methods.

    Example:
    In [1]: async def wrapper(nested_job):
       ...:     await nested_job()
    In [2]: async def job():
       ...:     pass

    In [3]: wrapper(job)
    Out[3]: <coroutine object wrapper at 0x104e04c48>

    In [4]: nested_job(wrapper, job)()
    Out[4]: <coroutine object wrapper --> job at 0x104f1d248>
    """
    if inspect.iscoroutinefunction(method):
        @wraps(method)
        async def impl(*args, **kwargs):
            return await method(nested, *args, **kwargs)
    else:
        @wraps(method)
        def impl(*args, **kwargs):
            return method(nested, *args, **kwargs)

    impl.__qualname__ = f'{method.__qualname__} --> {nested.__qualname__}'
    return impl
