import logging
import os
from collections import namedtuple

import requests

log = logging.getLogger(__name__)


class TvmInternalError(RuntimeError):
    pass


class TvmCheckError(RuntimeError):
    pass


class TicketInfo(namedtuple('TicketInfoBase', ('src', 'dst', 'scopes', 'debug_string', 'logging_string'))):

    @staticmethod
    def from_dict(data):
        return TicketInfo(
            src=int(data.get('src')),
            dst=int(data.get('dst')),
            scopes=data.get('scopes'),
            debug_string=data.get('debug_string'),
            logging_string=data.get('logging_string'),
        )


class Tvm(object):
    def __init__(self, tvm_daemon_url, client_id, local_token=None):
        self._client_id = client_id
        self._local_token = local_token or os.environ.get('QLOUD_TVM_TOKEN')
        self.tvm_daemon_url = tvm_daemon_url or 'http://localhost:1'
        self._checksrv_url = self.tvm_daemon_url + '/tvm/checksrv'
        self._get_tickets_url = self.tvm_daemon_url + '/tvm/tickets'

    def parse(self, service_ticket):
        """
        :type service_ticket: str
        :return: TicketInfo
        """
        try:
            response = requests.get(
                url=self._checksrv_url,
                params={
                    'dst': self._client_id,
                },
                headers={
                    'Authorization': self._local_token,
                    'X-Ya-Service-Ticket': service_ticket,
                },
            )
        except Exception as e:
            log.exception(e)
            raise TvmInternalError('TVM-tool internal communication problem: {}'.format(str(e)))
        if response.status_code != 200:
            log.error('TVM daemon status_code=%d, error: %s', response.status_code, response.text)
            try:
                response.json()['error']
            except:
                raise TvmInternalError('TVM daemon returned non-200: %d' % response.status_code)
        data = response.json()
        if 'error' in data:
            log.error('Error: %s ticket: %s', data['error'], data['logging_string'])
            raise TvmCheckError(data['error'])
        if int(data['dst']) != self._client_id:
            log.error('Wrong dst in ticket: %s', data['logging_string'])
            raise TvmCheckError('Wrong dst=%d in TVM ticket, should be %d', int(data['dst']), self._client_id)
        return TicketInfo.from_dict(data)

    def get(self, dst_client_id):
        """
        :type dst_client_id: int
        :return: str
        """
        try:
            response = requests.get(
                url=self._get_tickets_url,
                params={'src': self._client_id, 'dsts': dst_client_id},
                headers={'Authorization': self._local_token},
            )
        except Exception as e:
            log.exception(e)
            raise TvmInternalError(str(e))
        if response.status_code != 200:
            log.error('TVM daemon status_code=%d, error: %s', response.status_code, response.text)
            try:
                response.json()['error']
            except:
                raise TvmInternalError('TVM daemon returned non-200: %d; %s', response.status_code, response.text)
        data = response.json()
        if 'error' in data:
            log.error('Error: %s', data['error'])
            raise TvmCheckError(data['error'])
        if len(data) != 1:
            log.error('Single element in TVM response expected, got: %s', data)
            raise TvmInternalError('Single element in TVM response expected')
        ticket_data = list(data.values())[0]
        if 'error' in ticket_data:
            log.error('Error: %s ticket: %s', ticket_data['error'], ticket_data['logging_string'])
            raise TvmInternalError(ticket_data['error'])
        return ticket_data['ticket']
