require('dotenv').config();
let client = require('@yandex-int/st-client').init({token: process.env.TRACKER_TOKEN});

let request = require('requestretry');

function postStat(table, data) {
    return request.post({
        url: `https://upload.stat.yandex-team.ru/_api/report/data/Disk/${table}`,
        body: JSON.stringify(
            {
                scale: 'd',
                json_data: JSON.stringify({
                    "values": data
                })
            }),
        headers: {
            'Authorization': `OAuth ${process.env.STAT_TOKEN}`,
            'Content-Type': 'application/json'
        },
        maxAttempts: 2,   // (default) try 2 times
        retryDelay: 5000,  // (default) wait for 5s before trying again
        retryStrategy: request.RetryStrategies.HTTPOrNetworkError // (default) retry on 5xx or network errors
    })
}

const commits = {
    web: {
        critical: 3,
        blocker: 2
    },
    soft: {
        critical: 14,
        blocker: 3
    }
}

const projects = [
    //Web Телемост
    {
        project: 'Web_Telemost',
        type: 'web',
        queryOpened: 'Filter: 407114',
        queryClosed: 'Filter: 407115',
        queryCrits: 'Filter: 407134'
    },

    //Web Диск
    {
        project: 'Web_Disk',
        type: 'web',
        queryOpened: 'Filter: 407112',
        queryClosed: 'Filter: 407113',
        queryCrits: 'Filter: 407133'
    },

    //Desktop Software Telemost
    {
        project: 'Desktop_Soft_Telemost',
        type: 'soft',
        queryOpened: 'Filter: 407183',
        queryClosed: 'Filter: 407184',
        queryCrits: 'Filter: 407185'
    },

    //Desktop Software Disk
    {
        project: 'Desktop_Soft_Disk',
        type: 'soft',
        queryOpened: 'Filter: 407187',
        queryClosed: 'Filter: 407186',
        queryCrits: 'Filter: 407188'
    },
    //iOS Telemost
    {
        project: 'iOS_Telemost',
        type: 'soft',
        queryOpened: 'Filter: 407338',
        queryClosed: 'Filter: 407340',
        queryCrits: 'Filter: 407341'
    },

    //iOS Disk
    {
        project: 'iOS_Disk',
        type: 'soft',
        queryOpened: 'Filter: 407342',
        queryClosed: 'Filter: 407343',
        queryCrits: 'Filter: 407344'
    },
    //Android Telemost
    {
        project: 'Android_Telemost',
        type: 'soft',
        queryOpened: 'Filter: 407333',
        queryClosed: 'Filter: 407334',
        queryCrits: 'Filter: 407336'
    },

    //Android Disk
    {
        project: 'Android_Disk',
        type: 'soft',
        queryOpened: 'Filter: 407349',
        queryClosed: 'Filter: 407350',
        queryCrits: 'Filter: 407351'
    },
    //Web Adminka
    {
        project: 'Web_Adminka',
        type: 'web',
        queryOpened: 'Filter: 420950',
        queryClosed: 'Filter: 420946',
        queryCrits: 'Filter: 420966'
    },
];

/*
    internal function
 */
async function* historyChangesGenerator(issue) {
    let params = {
        type: ['IssueWorkflow']
    }

    let historyEntries = [];
    do {
        historyEntries = await client.issueChangelog(issue, params);
        for (let item of historyEntries) {
            params.id = item.id;
            yield item;
        }
    } while (historyEntries.length > 0)
}

/*
    internal function
 */
async function historyGetter(issue) {
    return new Promise(async (resolve) => {
            let closedAt;
            let historyGenerator = historyChangesGenerator(issue);

            for await (let historyChangeItem of historyGenerator) {
                historyChangeItem.fields.forEach(oneFieldChange => {
                        //console.log(oneFieldChange.field.id + ' @ ' + historyChangeItem.updatedAt.slice(0, 10));
                        if (oneFieldChange.field.id === 'status' && oneFieldChange.to.key === 'closed') {
                            closedAt = historyChangeItem.updatedAt;
                        }
                    }
                )
            }
            resolve(closedAt);
        }
    );
}


async function issuesCritsGetter(project) {
    const answer = {
        critical: [],
        blocker: []
    }

    function pushAnswer(issue, time) {
        const createdDay = new Date(issue.createdAt);
        const today = new Date(time);
        const daysDiff = Math.ceil((today - createdDay) / 1000 / 60 / 60 / 24);
        const priority = issue.priority.key;
        if (priority === 'critical' || priority === 'blocker') {
            if (daysDiff > commits[project.type][priority]) {
                answer[priority].push(issue.key);
            }
        }
    }

    return client.issues({
        query: project.queryCrits,
        //ожидаем, что критов и блокеров не более 100 в проекте. Если больше, то что-то не так с проектом =)
        perPage: 100
    }).then((issues) => {
        return Promise.all(issues.map(async (issue) => {
            if (issue.status.key !== 'closed') {
                pushAnswer(issue, Date.now())
            } else {
                await historyGetter(issue.key)
                    .then((time) => {
                        pushAnswer(issue, time)
                    })
            }
        }))
            .then(() => answer);
    })
}


function weightGetter(projects, type, page = 1, weight = 0) {
    return client.issues({
        query: projects[type],
        page: page,
        perPage: 100
    })
        .then(function (issues) {
            issues.forEach((issue) => {
                if (Number.isInteger(issue.weightOne) && issue.weightOne > 0) {
                    weight += Number(issue.weightOne);
                } else {
                    let addable;
                    switch (issue.priority.key) {
                        case 'blocker':
                            addable = 5000;
                            break;
                        case 'critical':
                            addable = 1500;
                            break;
                        case 'normal':
                            addable = 500;
                            break;
                        case 'minor':
                            addable = 65;
                            break;
                        case 'trivial':
                            addable = 15;
                            break;
                        default:
                            addable = 500;
                            break;
                    }
                    weight += addable;
                }
            })
            return issues;
        })
        .then((issues) => {
            //на случай ошибки в запросе -- больше 1000 тикетов не должно быть
            if (page > 10) {
                throw new Error('threshold for fixed pages reached')
            }
            if (issues.length > 0) {
                return weightGetter(projects, type, page + 1, weight)
            } else {
                return weight;
            }
        })
}

let date = new Date();
date = date.toISOString().slice(0, 10);

Promise.all(projects.map(async (project) => {
    return {
        fielddate: date,
        project: project.project,
        fixed: await weightGetter(project, 'queryClosed'),
        weight: await weightGetter(project, 'queryOpened')
    }
}))
    .then((result) => {
        console.log(result);
        postStat('maksimbark', result).then(res => console.log(res.body));
    })
    .then(() => {
        Promise.all(
            projects.map(
                project => issuesCritsGetter(project)
                    .then(res => {
                            const answerArr = [];
                            for (let type in res) {
                                res[type].forEach(item => {
                                    answerArr.push({
                                        fielddate: date,
                                        project: project.project,
                                        id: item,
                                        priority: type,
                                        count: 1
                                    })
                                })
                            }
                            return answerArr;
                        }
                    )
            ))
            .then(arr => {
                let answer = [];
                arr.forEach(item => {
                    answer = answer.concat(item)
                })
                if (answer.length > 0) {
                    console.log(answer);
                    postStat('disk-calls-zbp-crits', answer).then(res => console.log(res.body));
                }
            });
    });