import argparse
import datetime
import json
import os
from typing import Union, Any

import requests
import sys
import urllib3
from startrek_client import Startrek
from tp_api_client import tp_api_client
from set_secret import set_secret

urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)

CHECKBOX = '<# <input type="checkbox"> #>'


class Scenario:
    def __init__(self):
        self.testing_version = False
        self.ticket_text = False
        self.ticket_text_edit = False
        self.ticket_translate = False
        self.ticket_libraries = False
        self.ticket_react = False
        self.ticket_telemost = False
        self.version_st = False


def maintv(requester_mm, mobile_platform, mobileapp_version, mobile_dashboard, release_ticket_testpalm, release_notes, update_libs, android_beta):
    create = Scenario()
    create.version_st = True if release_ticket_testpalm == 'on' else False
    create.testing_version = True if release_ticket_testpalm == 'on' else False
    create.ticket_text = True if release_notes == 'on' else False
    create.ticket_libraries = True if update_libs == 'on' else False
    is_beta = True if android_beta == 'on' else False

    with open(os.path.join(os.path.dirname(__file__), f'variable_{mobile_platform}.json'), encoding='utf-8') as file:
        data = json.load(file)


    st = StarTrackReleaseManager(requester_mm, mobile_platform, mobileapp_version, mobile_dashboard)

    if create.version_st:
        st.create_version_startrek()

        st.create_ticket_for_testing(requester_mm, mobile_platform, mobile_dashboard)

        tp = TestPalm(data, st, is_beta)
        tp.create_testpalm_version(title='{}{}'.format(tp.platform, tp.version),
                                   description=st.description_for_testing_ticket)
        tp.create_first_runs()


    if create.ticket_libraries:
        if mobile_platform == 'iOS':
            st.create_ticket_in_our_queue(
                summary='Update 3rd party libraries',
                description='',
                checklist=data['update_libraries_checklist'])

    if create.ticket_text:
        st.create_ticket_in_our_queue \
            (summary='Ticket for release notes',
             description='**((https://st.yandex-team.ru/createTicket?queue=EDIT&_form=15297 Заказать текст для Release Notes))\n'
                         '((https://st.yandex-team.ru/createTicket?queue=TRANSLATE&_form=13067 Перевести текст Release Notes))**')


class StarTrackReleaseManager:
    def __init__(self, requester_mm, mobile_platform, mobileapp_version, mobile_dashboard):
        with open(os.path.join(os.path.dirname(__file__), f'variable_{mobile_platform}.json'), encoding='utf-8') as file:
            data = json.load(file)
        self.client = Startrek(base_url='https://st-api.yandex-team.ru',
                               useragent=requester_mm,
                               token=os.environ["STARTRECK_TOKEN"])
        self.token = os.environ["STARTRECK_TOKEN"]

        self.queue = str(data['queue'])
        self.beta = data['beta']
        self.release_type = str(data['release_type'])
        self.checklist = data['checklist']
        self.dashboard = data['dashboard_filters']

        self.platform = str(mobile_platform)
        self.version = str(mobileapp_version)

        self.id_version = self.get_id_version()
        self.description_for_testing_ticket = "No description"
        self.task_key = str(data['release_ticket']) if data['release_ticket'] else None

    def create_version_startrek(self):
        versions = self.client.queues[self.queue].versions
        versions_name_list = list(map(lambda x: x['name'], versions))
        if f'{self.platform} {self.version}' not in versions_name_list:
            version = self.client.versions.create(queue=self.queue, name='{} {}'.format(self.platform, self.version))
            self.id_version = version.id


    def get_description_for_testing_ticket(self, mobile_platform, mobile_dashboard):
        with open(os.path.join(os.path.dirname(__file__), f'variable_{mobile_platform}.json'), encoding='utf-8') as file:
            data_json = json.load(file)
        all_ticket = self.client.filters.create(name='{} {} - All tickets'.format(self.platform, self.version),
                                                query=data_json['all_ticket'].replace('[version]', self.version))

        found_bugs = self.client.filters.create(name='{} {} - Found bugs'.format(self.platform, self.version),
                                                query=data_json['found_bugs'].replace('[version]', self.version))

        backlog_bug = self.client.filters.create(name='{} {} - Backlog bugs'.format(self.platform, self.version),
                                                 query=data_json['backlog_bug'].replace('[version]', self.version))

        backlog_task = self.client.filters.create(name='{} {} - Backlog task'.format(self.platform, self.version),
                                                  query=data_json['backlog_task'].replace('[version]', self.version))

        url_testpalm = 'Version in TestPalm  {testpalm}/{platform}{version}'.format(testpalm=data_json['testpalm'],
                                                                                    version=self.version,
                                                                                    platform=self.platform)


        description = \
            'Link on dashboard {dashboard}\n\n' \
            'All tickets was fixed in version {version} https://st.yandex-team.ru/filters/filter:{all_ticket}\n' \
            'The bugs found while testing version {version} https://st.yandex-team.ru/filters/filter:{found_bugs}\n\n' \
            'Bugs from backlog was fixed in version {version} https://st.yandex-team.ru/filters/filter:{backlog_bug}\n' \
            'Tasks from backlog was fixed in version {version} https://st.yandex-team.ru/filters/filter:{backlog_task}\n\n' \
            '{url_testpalm}\n'.format(dashboard=mobile_dashboard, all_ticket=all_ticket.id,
                                           backlog_bug=backlog_bug.id, backlog_task=backlog_task.id,
                                           version=self.version, found_bugs=found_bugs.id,
                                           url_testpalm=url_testpalm)
        return description

    def get_id_version(self):
        all_version = self.client.queues[self.queue].versions
        for item in all_version:
            if item.name == '{} {}'.format(self.platform, self.version):
                self.id_version = item.id
                return self.id_version
        return ''

    def get_id_component(self):
        all_components = self.client.queues[self.queue].components
        for item in all_components:
            if item.name == '{}'.format(self.platform):
                return item.id
        return ''

    def create_checklist(self, key, checklist):
        headers_startrek = {
            'Authorization': f'OAuth {self.token}',
            'Content-Type': 'application/json',
            'User-Agent': 'curl/7.53.1',
            'Connection': 'close',
            'Accept': '*/*'}

        data = [{"checked": False, "text": item} for item in list(checklist)]

        r = requests.post(f'https://st-api.yandex-team.ru/v2/issues/{key}/checklistItems',
                          data=json.dumps(data),
                          headers=headers_startrek)

    def create_ticket_for_testing(self, requester_mm, mobile_platform, mobile_dashboard):
        with open(os.path.join(os.path.dirname(__file__), f'variable_{mobile_platform}.json'), encoding='utf-8') as file:
            data = json.load(file)
        release_type = str(data['release_type']) if self.release_type else self.release_type
        self.description_for_testing_ticket = self.get_description_for_testing_ticket(mobile_platform, mobile_dashboard)
        id_component = self.get_id_component()

        ticket = self.client.issues.create(queue=self.queue,
                                           summary='{} {} testing'.format(self.platform, self.version),
                                           description=self.description_for_testing_ticket,
                                           type='release',
                                           affectedVersions=self.id_version,
                                           fixVersions=self.id_version,
                                           components=id_component,
                                           releaseType=release_type,
                                           qa=requester_mm)
        self.task_key = ticket.key
        self.create_checklist(self.task_key, self.checklist)

    def create_task_texts(self, summary, description, queue):
        task = self.client.issues.create(queue=queue,
                                         summary=summary,
                                         description=description,
                                         type='task')

        testing_ticket = self.client.issues[self.task_key]
        testing_ticket.comments.create(text='Create ticket for release notes: {}'.format(task.key))

    def create_ticket_in_our_queue(self, summary, description, checklist=None):
        id_version = self.get_id_version()
        id_component = self.get_id_component()
        task_key = self.client.issues.create(queue=self.queue,
                                             summary="[{}] {}".format(self.version, summary),
                                             description=description,
                                             type='task',
                                             affectedVersions=id_version,
                                             fixVersions=id_version,
                                             components=id_component)
        if checklist is not None:
            self.create_checklist(task_key.key, checklist)
        testing_ticket = self.client.issues[self.task_key]
        testing_ticket.comments.create(text='Create ticket: {}'.format(task_key.key))


class TestPalm:
    def __init__(self, data, st, is_beta):

        self.project = data['testpalm_project']
        self.client = tp_api_client.TestPalmClient(auth=os.environ["TESTPALM_OAUTH"] )
        self.platform = st.platform
        self.version = st.version
        self.ticket_for_testing = st.task_key
        self.queue = data['queue']
        self.suites = data['suites_beta'] if is_beta else data['suites']

    def create_testpalm_version(self, title, description):
        data = {
            'id': title,
            'description': description
        }

        self.client.create_version(project=self.project, data=data)

    def create_first_runs(self):
        for item in self.suites:
            self.create_testrun(item, '{version} {name} {enviroment}'.format(version=self.version,
                                                                             name=item['run'],
                                                                             enviroment=item['enviroment'].strip()))

    def create_testrun(self, suite, title):
        data = {
            'title': title,
            'version': '%s%s' % (self.platform, self.version),
            'environments': [{
                'title': suite['enviroment'],
                'description': suite['enviroment'],
                'default': False
            }],
            'testSuite': {
                'id': str(suite['key'])
            },
            'parentIssue': {
                'assignee': [],
                'createdTime': 0,
                'display': self.ticket_for_testing,
                'groupId': self.queue,
                'id': self.ticket_for_testing,
                'isBug': False,
                'url': 'https://st.yandex-team.ru/{}'.format(self.ticket_for_testing),
                'status': {
                    'id': 'open',
                    'title': 'Открыт'
                },
                'isResolved': False,
                'title': self.queue,
                'trackerId': 'Startrek'
            }
        }
        self.client.create_testrun_from_suite(project=self.project, data=data)





if __name__ == '__main__':
    set_secret.set_secrets()
