import json
import logging
import os
import ssl
import sys
import traceback
import urllib.parse
from datetime import datetime, timedelta
from enum import Enum
from typing import Optional, Dict, List

import requests

from set_secret import set_secret

FORMAT = '%(asctime)s:%(levelname)s:%(name)s:%(message)s'
logging.basicConfig(level=logging.INFO, format=FORMAT)
ssl._create_default_https_context = ssl._create_unverified_context
logging.getLogger('booking')

# for more information about this file see https://booking.yandex-team.ru/#/help?section=api

# name of quota. can be found by https://booking.yandex-team.ru/#/help?section=api-quotas
QUOTA_CODE: str = "qs_testpalm_separate_mail-liza_ru"
BASE_URL: str = "https://booking.yandex-team.ru/"
# max acceptable quota (in working hours) or None if does not matter
MAX_QUOTA: Optional[float] = None  # todo: enter your max quota or None here
# do not change
PRICE_PER_WH: float = 120.0
# max acceptable duration of task in hours
MAX_DURATION: Optional[float] = None  # todo: enter your max duration for booking
# max ratio between actual duration and MAX_DURATION. 1.2 means 120% of MAX_DURATION is ok but no more
DURATION_MAX_RATIO: float = 1.2  # todo: enter your appropriate ratio here


# type of speed
# slow is for bookings with low priority
# normal - for everyday bookings
# urgent - for extremely fast bookings with short deadline (higher costs)
class Speed(Enum):
    SLOW = 0
    NORMAL = 1
    URGENT = 2

    def lower(self):
        if self == Speed.URGENT:
            return Speed.NORMAL
        elif self == Speed.NORMAL:
            return Speed.SLOW
        else:
            raise Exception("The speed mode is already the slowest")

    def higher(self):
        if self == Speed.SLOW:
            return Speed.NORMAL
        elif self == Speed.NORMAL:
            return Speed.URGENT
        else:
            raise Exception("The speed mode is already the highest possible.")


def get_next_range(start_ts: int):
    for item in get_acceptable_time_range():
        if item["startTs"] > start_ts:
            return item
    return None


def get_time_range(start_ts: int):
    for item in get_acceptable_time_range():
        if item["startTs"] == start_ts:
            return item


def get_acceptable_time_range():
    return [
        {
            "startTs": int(datetime.now().timestamp() * 1000),
            "endTs": int((datetime.now() + timedelta(days=5)).timestamp() * 1000),
        }
    ]


def sort_time_ranges(time_ranges: List):
    return sorted(time_ranges, key=lambda k: k["startTs"], reverse=True)


def validate_time_ranges(time_ranges: List):
    for time_range in time_ranges:
        if time_range["startTs"] >= time_range["endTs"]:
            raise Exception(
                "Time range has endTs "
                + str(time_range["endTs"])
                + " is less than startTs "
                + str(time_range["startTs"])
            )
    if len(time_ranges) == 0:
        raise Exception("Time ranges cannot be empty")


def create_booking(title: str, params: Dict, estimation: Dict, start_early=False):
    url = BASE_URL + "api/bookings/assessor-testing/default"
    booking_data = {
        "title": title,
        "params": params,
        "estimate": estimation,
    }
    if start_early:
        booking_data['metaData'] = {
            'customData': {'allowStartBeforeBookingStartTs': True},
            'subscribers': []
        }

    logging.info("Attempt to create: %s", booking_data)
    response = requests.post(
        url,
        data=json.dumps(booking_data),
        headers={
            "Authorization": "OAuth " + os.environ["HITMAN_TOKEN"],
            "Content-Type": "application/json",
            "Accept": "application/json",
        },
        verify=False,
        timeout=40,
    )
    if response.status_code == 200:
        logging.info("Created booking: %s", response.json())
        return response.json()
    else:
        logging.error(response.request.url)
        logging.error(response.status_code)
        logging.error(response.text)
        return None


def check_duration(body: Dict, estimate_result: Dict):
    logging.info("Checking duration...")
    current_duration = (
        (estimate_result["deadlineTs"] - estimate_result["startTs"])
        / 1000.0
        / 60.0
        / 60.0
    )

    if (
            MAX_DURATION is not None
            and MAX_DURATION != 0
            and current_duration / MAX_DURATION > DURATION_MAX_RATIO
    ):
        logging.info(
            "Duration is %s with MAX_DURATION %s", current_duration, MAX_DURATION
        )
        current_speed = Speed[body["speedMode"]].higher()
        logging.info(
            "Speed mode was changed from %s to %s",
            body["speedMode"],
            current_speed.name,
        )
        body["speedMode"] = current_speed.name
        return False, body
    else:
        logging.info("Duration is ok!")
        return True, body


def check_max_quota(body: Dict, estimate_result: Dict):
    logging.info("Checking max quota...")
    if MAX_QUOTA is not None and PRICE_PER_WH * MAX_QUOTA < estimate_result["minQuota"]:
        diff_quota = MAX_QUOTA * PRICE_PER_WH - estimate_result["minQuota"]
        logging.info("Quota is more than %s", abs(diff_quota))
        current_speed = Speed[body["speedMode"]].lower()
        logging.info(
            "Speed mode was changed from %s to %s",
            body["speedMode"],
            current_speed.name,
        )
        body["speedMode"] = current_speed.name
        return False, body
    else:
        logging.info("Max quota is ok!")
        return True, body


def check_time_range(body: Dict, estimate_result: Dict):
    logging.info("Checking start time...")
    time_range = get_acceptable_time_range()[0]
    if (
            estimate_result["startTs"] > time_range["endTs"]
    ):
        logging.info("StartsTs was moved to %s", estimate_result["startTs"])
        next_time_range = get_next_range(body["startTsFrom"])
        if next_time_range is None:
            raise Exception("No acceptable time periods!")
        body["startTsFrom"] = next_time_range["startTs"]
        logging.info("Changed startTs is %s", body["startTsFrom"])
        return False, body
    else:
        logging.info("StartTs is ok!")
        return True, body


def estimate(title: str, body: Dict, start_early=False):
    """
    tries to estimate time and checks with parameters given
    checks with following logic:
    1. first tries to booking in the beginning of the earliest time period.
    2. if start time is not in the earliest time range, tries to booking to the next time range or throws Exception
    3. if start time is ok, it checks task duration. If MAX_DURATION is set and duration is more than max,
    the speed mode is increased. If speed mode is already the highest, then exception is raised.
    4. If duration is ok, it checks max quota. If MAX_QUOTA is set and is more than max, it tries to set lower speed.
    Id the speed mode is the lowest, then it raises exception.
    5. If all parameters are ok, then the booking is created.
    :param title: name of the booking
    :param body: parameters to estimate according to type
    :param start_early: parameter to run process earlier if possible
    :return: booking or None
    """
    logging.info(f'Estimating {body}')
    url = BASE_URL + "api/bookings/assessor-testing/default/estimate"
    response = requests.post(
        url,
        data=json.dumps(body),
        headers={
            "Authorization": "OAuth " + os.environ["HITMAN_TOKEN"],
            "Content-Type": "application/json",
            "Accept": "application/json",
        },
        verify=False,
        timeout=40,
    )
    if response.status_code != 200:
        logging.error(response.status_code)
        logging.error(response.text)
        return None

    estimate_result = response.json()
    logging.info("Estimated result %s", estimate_result)

    is_time_ok, body = check_time_range(body, estimate_result)

    if not is_time_ok:
        estimate(title, body)
    else:
        is_duration_ok, estimate_request = check_duration(body, estimate_result)
        if not is_duration_ok:
            estimate(title, estimate_request)
        else:
            is_quota_ok, estimate_request = check_max_quota(body, estimate_result)
            if not is_quota_ok:
                estimate(title, estimate_request)
            else:
                print('can create!')
                return create_booking(title, body, estimate_result, start_early)


def book_custom_volume(
        title: str,
        volume_amount: float,
        environment_distribution: Dict,
        speed: Enum,
        start_ts: int,
):
    """

    :param title: name of the booking
    :param volume_amount: additional volume for the booking in БО
    :param environment_distribution: distribution of environments that would be used with rate.
        Summary of rate should be equal to 1. Example,  {"64": 0.2, "63": 0.8}.
        Key is code for environment and value is for rate.
        More on https://booking.yandex-team.ru/#/help?section=api-envs
    :param speed: type of speed mode. NORMAL is recommended
    :param start_ts: preferred start time in milliseconds
    :return: created booking or None
    """
    body = {
        "quotaSource": QUOTA_CODE,
        "speedMode": speed.name,
        "volumeDescription": {
            "volumeSources": [],
            "customVolume": {
                "amount": volume_amount,
                "environmentDistribution": environment_distribution,
            },
        },
        "startTsFrom": start_ts,
    }
    return estimate(title, body)


def book_st_filter(
        title: str, st_filter: str, speed: Enum, custom_volume: float, start_ts: int
):
    """

    :param title: name of the booking
    :param st_filter: link to Startrek filter.Example, "https://st.yandex-team.ru/issues/29390"
    :param speed: type of speed mode. NORMAL is recommended
    :param custom_volume: additional volume for the booking in БО
    :param start_ts: preferred start time in milliseconds
    :return: created booking or None
    """
    body = {
        "quotaSource": QUOTA_CODE,
        "speedMode": speed.name,
        "volumeDescription": {
            "volumeSources": [{"type": "ST_TICKET_FILTER", "values": [st_filter]}],
            "customVolume": custom_volume if custom_volume > 0 else None,
        },
        "startTsFrom": start_ts,
    }
    return estimate(title, body)


def book_st_tickets(
        title: str, st_tickets: List, speed: Enum, custom_volume: float, start_ts: int
):
    """

    :param title: name of the booking
    :param st_tickets: array of links to Startrek tickets.
    Example, ["https://st.yandex-team.ru/TESTING-1", "https://st.yandex-team.ru/TESTING-2"]
    :param speed: type of speed mode. NORMAL is recommended
    :param custom_volume: additional volume for the booking in БО
    :param start_ts: preferred start time in milliseconds
    :return: created booking or None
    """
    body = {
        "quotaSource": QUOTA_CODE,
        "speedMode": speed.name,
        "volumeDescription": {
            "volumeSources": [{"type": "ST_TICKET", "values": st_tickets}],
            "customVolume": custom_volume if custom_volume > 0 else None,
        },
        "startTsFrom": start_ts,
    }
    return estimate(title, body)


def book_testpalm_version(
        title: str, quota: str, versions: List, speed: Enum, custom_volume: float, start_ts: int, start_early=False
):
    """

    :param title: name of the booking
    :param quota: quota for run
    :param versions:  array of links to Testpalm versions.
    Example, ["https://testpalm.yandex-team.ru/test/version/v_1", "https://testpalm.yandex-team.ru/test/version/v_2"]
    :param speed: type of speed mode. NORMAL is recommended
    :param custom_volume: additional volume for the booking in БО
    :param start_ts: preferred start time in milliseconds
    :return: created booking or None
    """
    body = {
        "quotaSource": quota,
        "speedMode": speed.name,
        "volumeDescription": {
            "volumeSources": [{"type": "TEST_PALM", "values": versions}],
            "customVolume": custom_volume if custom_volume > 0 else None,
        },
        "startTsFrom": start_ts,
    }
    return estimate(title, body, start_early)


def get_start_time():
    """
    Sorts and validates time ranges
    If all valid, returns startTs of first time range
    :return:
    """
    acceptable_time_ranges = sort_time_ranges(get_acceptable_time_range())
    validate_time_ranges(acceptable_time_ranges)
    return acceptable_time_ranges[0]["startTs"]


def get_closest_free_place(hours):
    body = {
        "quotaSource": QUOTA_CODE,
        "speedMode": Speed.NORMAL.name,
        "volumeDescription": {
            "volumeSources": [],
            "customVolume": {
                "amount": hours,
                "environmentDistribution": {'50': 0.25, '49': 0.25, '45': 0.25, '48': 0.25},
            },
        },
        "startTsFrom": get_start_time(),
    }

    logging.info(f'Estimating {body}')
    url = BASE_URL + "api/bookings/assessor-testing/default/estimate"
    response = requests.post(
        url,
        data=json.dumps(body),
        headers={
            "Authorization": "OAuth " + os.environ["HITMAN_TOKEN"],
            "Content-Type": "application/json",
            "Accept": "application/json",
        },
        verify=False,
        timeout=40,
    )
    if response.status_code != 200:
        logging.error(response.status_code)
        logging.error(response.text)
        logging.error(response.content)
        return None

    estimate_result = response.json()
    return datetime.fromtimestamp(estimate_result['startTs'] // 1000)


def book_testpalm_version_by_name(quota, testpalm_project, testpalm_version_name, start_early=False):
    """Делает бронь в бронировщике и возвращает booking id для передачи в hitman"""
    url_version_name = urllib.parse.quote(testpalm_version_name)
    testpalm_url = f'https://testpalm.yandex-team.ru/{testpalm_project}/version/{url_version_name}'
    try:
        return book_testpalm_version(
            f'Бронь для регресса {testpalm_version_name}',
            quota,
            [testpalm_url],
            Speed.NORMAL,
            0,
            get_start_time(),
            start_early
        )['bookingId']
    except Exception as e:
        traceback.print_exception(*sys.exc_info())
        logging.error(e)


if __name__ == '__main__':
    set_secret.set_secrets()
    print(book_testpalm_version_by_name('qs_testpalm_separate_cal_ru', 'cal', 'v1.81 14.08.20 16.23.12 Асессоры pub'))
