import logging
import os
import sys
import time
import traceback

import requests
from retrying import retry

dir_path = os.path.dirname(os.path.realpath(__file__))
sys.path.insert(0, dir_path)
sys.path.insert(0, os.path.split(dir_path)[0])
from assessors_run.assessors_queue import AssessorsRun, AssessorStatus, BookingMode

from project_configs import booking_quota, runs, startrack_components
from api_classes.booking import book_testpalm_version_by_name
from api_classes.hitman import Hitman
from api_classes.startrack import ST
from api_classes.testpalm import TestPalm, RunCreationMode
from set_secret import set_secret

FORMAT = '%(asctime)s:%(levelname)s:%(name)s:%(message)s'
logging.basicConfig(level=logging.INFO, format=FORMAT)


@retry(wait_fixed=5000, stop_max_attempt_number=3)
def write_to_nirvana(data_props, requester):
    headers = {
        'Content-Type': 'application/json',
        'Authorization': 'OAuth %s' % os.environ["HITMAN_TOKEN"]
    }

    start_proc_url = 'https://hitman.yandex-team.ru/api/v1/execution/start/testung_mail_autostart'

    data = {
        'requester': requester,
        'properties': data_props
    }

    resp = requests.post(start_proc_url, verify=False, headers=headers, json=data)
    logging.info(resp.text)


def make_run(run: AssessorsRun):
    try:
        set_secret.set_secrets()
        run.status = AssessorStatus.started

        version_name = run.version_name
        tp_project = run.tp_project
        special_condition = run.special_condition
        test_stand = run.test_stand
        requester = run.requester
        version_config_name = run.version_config_name
        mode = run.mode
        filter_tp = run.filter_tp
        environment = run.environment

        if version_config_name == '':
            version_config_name = 'custom'

        startreck_version_name = f'{version_name} {time.strftime("%d.%m.%y %H.%M.%S")}'.strip()
        testpalm_version_name = f'{version_name} {time.strftime("%d.%m.%y %H.%M.%S")} Асессоры {version_config_name}'.strip()

        version = next(filter(lambda x: x['version'] == version_config_name, runs[tp_project]))

        if mode.value == RunCreationMode.GET_FROM_CONFIG.value:
            run_mode = RunCreationMode(version['run_mode'])
        else:
            run_mode = mode

        if len(environment) == 0:
            environment = version['environment']

        run.status = AssessorStatus.startrek
        startreck_queue_client = ST('MAILEXP')
        version_id = startreck_queue_client.create_version(startreck_version_name)
        task_key = startreck_queue_client.create_task_for_testing(
            f'Протестировать асессорами релиз {startreck_version_name}',
            '',
            version_id,
            tp_project,
            startrack_components[tp_project]
        )

        run.status = AssessorStatus.testpalm
        testpalm_project_client = TestPalm(tp_project)
        testpalm_project_client.create_version_tp(
            testpalm_version_name,
            test_stand,
            requester
        )
        testpalm_project_client.set_run_creation_mode(run_mode)
        if filter_tp:
            testpalm_project_client.make_runs_from_filter(
                filter_tp,
                environment,
                testpalm_version_name,
            )
        else:
            testpalm_project_client.make_runs_from_suites(
                version['suites'],
                environment,
                testpalm_version_name,
                task_key=task_key
            )

        run.status = AssessorStatus.booking

        start_early = True if run.run_earlier else False
        booking_id = 0
        booking_regular_id = 0
        if run.booking_mode.value == BookingMode.regular.value:
            booking_regular_id = version['booking_uuid']
        elif run.booking_mode.value == BookingMode.new.value:
            booking_id = book_testpalm_version_by_name(
                booking_quota[tp_project], tp_project, testpalm_version_name, start_early)
        elif run.booking_mode.value == BookingMode.custom.value:
            booking_id = run.booking_id.strip()
        logging.info(f'Booking id: {booking_id}, booking regular id: {booking_regular_id}')

        run.status = AssessorStatus.hitman
        hitman_client = Hitman(version['hitman_process'])
        hitman_client.send_data_to_hitman(
            requester,
            testpalm_version_name,
            special_condition,
            test_stand,
            booking_id,
            booking_regular_id,
            startreck_version_name,
            TestPalm.parse_version(testpalm_version_name)
        )

        run.status = AssessorStatus.finished
    except Exception:
        run.status = AssessorStatus.failed
        logging.error(traceback.format_exc())
        return


if __name__ == '__main__':
    set_secret.set_secrets()
    # version = next(filter(lambda x: x['version'] == 'Тач', runs['mail-touch']))
    # booking_id = book_testpalm_version_by_name(booking_quota['mail-touch'], 'mail-touch', '10.27 10.06.20 18.56.10 Асессоры Тач')
    # logging.info(f'Booking id: {booking_id}')
    #
    # hitman_client = Hitman(version['hitman_process'])
    # hitman_client.send_data_to_hitman(
    #     'a-zoshchuk',
    #     '10.27 10.06.20 18.56.10 Асессоры Тач',
    #     '10.27.1 или 10.27.2',
    #     'ub7-qa.mail.yandex.ru',
    #     booking_id
    # )
