import codecs
import json
import logging
import os
import re
import sys
import traceback
from time import time, sleep

import requests
import urllib3

scriptPath = os.path.dirname(os.path.abspath(__file__))
sys.path.insert(0, os.path.split(scriptPath)[0])

from urllib.parse import quote
from utils.utils import Service
from LogParse.client_part import logParser
from autotest_run.parser_touch import find_unique_errors, find_unnormal_errors_amount
from retrying import retry
from utils.send_message import send_message

from utils.config import Constants
from utils.sync_data import sync_data

FORMAT = '%(asctime)s:%(levelname)s:%(name)s:%(message)s'
logging.basicConfig(level=logging.INFO, format=FORMAT)
log = logging.getLogger("nightlyAutotestRunner")
urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)
logging.getLogger("requests.packages.urllib3.connectionpool").setLevel(logging.WARNING)

running_test_statuses = ['RUNNING', 'RUNNABLE']
finished_run_statuses = ['REPORT_READY', 'REPORT_FAILED']

retries = {
    Service.liza.value: 3,
    Service.touch.value: 2,
    Service.cal.value: 3
}

to = ['a-zoshchuk@yandex-team.ru',
      'cosmopanda@yandex-team.ru', 'marchart@yandex-team.ru', 'sbdsh@yandex-team.ru',
      'crafty@yandex-team.ru', 'sshelgunova@yandex-team.ru', 'mariya-murm@yandex-team.ru',
      'oleshko@yandex-team.ru'
      ]

subj_autotests = 'Отчет по ночным автотестам'
subj_grep = 'Отчет по ночному грепу логов'
body = ''
ticket_url_part = 'https://st.yandex-team.ru/'


@retry(stop_max_attempt_number=3, wait_fixed=10000)
def check_if_run_finished(launch_id):
    headers = {
        "Content-Type": "application/json"
    }
    launch_status = 'RUNNABLE'
    start_time = time()
    while launch_status in running_test_statuses:
        sleep(20)
        r = requests.get(
            'http://aqua.yandex-team.ru/aqua-api/services/launch/%s' % launch_id,
            headers=headers,
            verify=False,
        )
        launch_status = r.json()["launchStatus"]
        if time() - start_time > 45 * 60:
            log.warning("Test with id %s is running for too long! Breaking" % launch_id)
            requests.delete(
                'http://aqua.yandex-team.ru/aqua-api/services/launch/%s' % launch_id,
                headers=headers,
                verify=False,
            )
            break
    log.info(launch_status + ' ' + launch_id)
    return r


def get_hosts(service):
    hosts = []
    path_to_file = Constants().nightly_values_local_file
    try:
        with codecs.open(path_to_file, 'r', encoding='utf-8') as f:
            tasks = json.load(f)
        for task in tasks:
            if 'stand' in task and service in task['service']:
                hosts.append(task['stand'])
    except ValueError:
        log.error('no json found, setting empty')
        log.error(traceback.format_exc())
    print(hosts)
    return hosts


def get_runs_data(service):
    data = []
    path_to_file = Constants().nightly_values_local_file
    try:
        with codecs.open(path_to_file, 'r', encoding='utf-8') as f:
            tasks = json.load(f)
        for task in tasks:
            if service in task['service']:
                data.append({'stand': task['stand'], 'corp_stand': task['corp_stand']})
    except ValueError:
        log.error('no json found, setting empty')
        log.error(traceback.format_exc())
    print(data)
    return data


def get_emails(data, service):
    emails = []
    path_to_file = Constants().nightly_values_local_file
    try:
        with codecs.open(path_to_file, 'r', encoding='utf-8') as f:
            tasks = json.load(f)
        for task in tasks:
            if ((task.get('stand', '') == data.get('stand', '')) and (task.get('corp_stand', '') == data.get('corp_stand', ''))):
                if service in task['service']:
                    emails = task['emails']
    except ValueError:
        log.error('no json found, setting empty')
        log.error(traceback.format_exc())
    return emails


def run_grep():
    data = get_runs_data(Service.liza.value)
    for task in data:
        try:
            emails = get_emails(task, Service.liza.value)
            if task.get('stand', '') == 'mail.yandex.ru':
                continue
            to_grep = 'a.zoshchuk@yandex-team.com'
            message_log = logParser.grep_2d_logs_for_host(task.get('stand', ''))
            message_log = '\n'.join(message_log)
            log.info('Send message:\n%s' % message_log)
            emails.append(to_grep)
            emails.append('mail-auto-reports@yandex-team.ru')
            send_message(emails, subj_grep, message_log)
        except Exception as e:
            log.error(traceback.format_exc())


def run_grep_with_mail(stand, service, emails):
    if service == Service.liza.value:
        to_grep = 'a.zoshchuk@yandex-team.com'
        message_log = logParser.grep_2d_logs_for_host(stand)
        message_log = '\n'.join(message_log)
        print('Send message:\n%s' % message_log)
        emails.append(to_grep)
        emails.append('mail-auto-reports@yandex-team.ru')
        send_message(emails, 'Отчёт по грепу логов', message_log)


def send_message_with_booster_urls(stand, service, emails, ticket):
    if 'http' in stand:
        stand = re.match('http[s]?://([^/]*)', stand).group(1)
    if service != Service.cal.value:
        message = []
        url_to_compare = get_compare_url(stand, service)

        message.append(f'Греп логов на стенде {stand}, сравнение с {url_to_compare}')
        message.append(
            f'<a href="{get_url_for_unique_errors(stand, service, url_to_compare)}">Уникальные ошибки (сортировать по дельте)</a>')
        message.append(
            f'<a href="{get_url_for_diff(stand, service, url_to_compare)}">Дифф ошибок (сортировать по дельте в процентах)</a>')
        message = '<br>'.join(message)
        print('Send message to mail:\n%s' % message)
        send_message(emails, 'Ссылки на логи', message)

        if ticket != '':
            message = []
            message.append(f'Греп логов на стенде {stand}, сравнение с {url_to_compare}')
            message.append(
                f'(({get_url_for_unique_errors(stand, service, url_to_compare)} Уникальные ошибки (сортировать по дельте) ))')
            message.append(
                f'(({get_url_for_diff(stand, service, url_to_compare)} Дифф ошибок (сортировать по дельте в процентах) ))')
            message = '\n'.join(message)
            print('Send message to startreck:\n%s' % message)
            send_startreck_comment(ticket, message)


def get_compare_url(stand, service):
    r = requests.get(
        f'https://api.error.yandex-team.ru/api/suggest?baseFilter=%7B%22project%22%3A%22{service}%22%2C%22period%22%3A%22month%22%7D&field=domain&like={service}-rc&service=error&table=errors',
        headers={'Authorization': f'OAuth {os.environ["ERR_BOOSTER_TOKEN"]}'},
        verify=False
    )
    print(r.text)
    prev_stand_versions = [re.findall('[0-9]+', host['key']) for host in r.json()]
    stand_version = re.findall('[0-9]+', stand)
    if len(stand_version) == 0:
        return 'mail.yandex.ru'
    major_version = int(stand_version[0])
    minor_version = int(stand_version[1])
    closest_version = (0, 0)
    for prev_stand_version in prev_stand_versions:
        major_prev_version = int(prev_stand_version[0])
        minor_prev_version = int(prev_stand_version[1])
        if major_version == major_prev_version:
            if minor_version == minor_prev_version + 1:
                return f'{service}-rc-{major_prev_version}-{minor_prev_version}.qa.mail.yandex.ru'
        if major_version > major_prev_version:
            if major_prev_version > closest_version[0]:
                closest_version = (major_prev_version, minor_prev_version)
            if (major_prev_version == closest_version[0]) & (minor_prev_version > closest_version[1]):
                closest_version = (major_prev_version, minor_prev_version)
    if closest_version[0] != 0:
        return f'{service}-rc-{closest_version[0]}-{closest_version[1]}.qa.mail.yandex.ru'
    return 'mail.yandex.ru'


def get_url_for_unique_errors(stand, service, compare_url):
    return f'https://error.yandex-team.ru/projects/{service}/compareJoin?' + \
           quote(
               'componentSettings={\"comparejoin\":{\"dimension\":\"domain\",\"leftValue\":\"' + \
               stand + \
               '\",\"rightValue\":\"' + \
               compare_url + \
               '\",\"joinType\":\"leftExcludingJoin\"}}' + \
               '&filter=message notLike %abook-contacts%' + \
               '&period=month',
               safe="=&"
           )


def get_url_for_diff(stand, service, compare_url):
    return f'https://error.yandex-team.ru/projects/{service}/compareJoin?' + \
           quote(
               'componentSettings={\"comparejoin\":{\"dimension\":\"domain\",\"leftValue\":\"' + \
               stand + \
               '\",\"rightValue\":\"' + \
               compare_url + \
               '\",\"joinType\":\"innerJoin\"}}' + \
               '&filter=message notLike %abook-contacts%' + \
               '&period=month',
               safe="=&"
           )


def get_browser_or_device(r, pack):
    browser_or_device = ''
    if 'browsers' in pack[1]:
        browser_or_device = next(
            (bro['value'] for bro in r.json()['pack']['properties'] if bro['key'] == 'webdriver.driver'),
            ''
        )
    if 'devices' in pack[1]:
        browser_or_device = next(
            (device['value'] for device in r.json()['pack']['properties'] if device['key'] == 'deviceType'),
            ''
        )
    return browser_or_device


def run_grep_touch():
    hosts = list(set(get_hosts(Service.touch.value)) - set(get_hosts(Service.liza.value)) - set(get_hosts(Service.cal.value)))
    for host in hosts:
        try:
            emails = get_emails(host, Service.touch.value)
            if host == 'mail.yandex.ru':
                continue
            to_grep = 'oleshko@yandex-team.ru'
            yesterday_new_errors, today_new_errors = find_unique_errors(host)
            unnormal_errors = find_unnormal_errors_amount(host)
            if yesterday_new_errors == {}:
                yesterday_new_errors = "Nothing is new yesterday\n"
            if today_new_errors == {}:
                today_new_errors = "\nNothing is new today"
            if unnormal_errors == {}:
                unnormal_errors = "Nothing is suspecious\n"
            message = 'Host: %s <br><br>Yesterday new errors:<br><br> %s <br><br>Today new errors:<br><br> %s ' \
                      '<br><br>Yesterday suspecious errors:<br><br> %s' % (
                          host, yesterday_new_errors, today_new_errors, unnormal_errors
                      )
            log.info('Send message:\n%s' % message)
            emails.append(to_grep)
            send_message(emails, subj_grep, message)
        except Exception as e:
            log.error("Something went wrong, exception:\n %s" % e)


def get_packs_by_category(packs, category):
    print(category)
    packs_to_run = dict(packs)
    if category == ['']:
        packs_ids = list(filter(lambda pack: not packs_to_run[pack].get('only_for_categories', False), packs_to_run))
        return {id: packs_to_run[id] for id in packs_ids}
    packs_ids = list(filter(lambda pack: category in packs_to_run[pack].get('category', []), packs_to_run))
    # запускаем скриночный смоук только в FF, если гоняем только категорию smoke
    if category == 'smoke':
        if '5d1e17788a904a4421895ab0' in packs_ids:
            packs_to_run['5d1e17788a904a4421895ab0']['browsers'] = ['chrome']

    print({id: packs_to_run[id] for id in packs_ids})
    return {id: packs_to_run[id] for id in packs_ids}


def send_startreck_comment(ticket, text):
    headers = {
        'Content-Type': 'application/json',
        'Authorization': 'OAuth %s' % os.environ['STARTRECK_TOKEN']
    }

    ticket_info = requests.get(
        "https://st-api.yandex-team.ru/v2/issues/%s" % ticket,
        headers=headers
    )
    print(ticket_info.json()['assignee']['id'])
    is_all_green = text.count('<font color="red">') == 0
    text = re.sub(r'https://st.yandex-team.ru/[A-Z\-0-9]*<br>', '', text)
    text = text.replace('<br>', '\n')
    text = re.sub(r'<font color="red">([^>]*)</font>', r'!!\1!!', text)
    text = re.sub(r'<font color="green">([^>]*)</font>', r'!!(зел)\1!!', text)

    data = {
        'text': text,
    }
    if not is_all_green:
        data['summonees'] = [ticket_info.json()['assignee']['id']]
    print(data)
    r = requests.post(
        "https://st-api.yandex-team.ru/v2/issues/%s/comments" % ticket,
        headers=headers,
        json=data
    )
    log.info(r.json())


def get_status(x):
    search_result = re.search('FAIL|SUCCESS', x)
    if search_result:
        return x[search_result.start()]
    else:
        return x


def get_sync_nightly_values():
    path_to_file = Constants().nightly_values_local_file
    log.info(f'Синкаем значения для ночных автотестов: {path_to_file}')
    data = []
    last_update = 0
    last_server_update = 0
    try:
        with codecs.open(path_to_file, 'r', encoding='utf-8') as f:
            raw_data = json.load(f)
        data = raw_data.get('data', [])
        last_update = raw_data.get('last_update', [])
        last_server_update = raw_data.get('last_server_update', [])
    except ValueError as e:
        log.error('No json found, setting empty')
        log.error(traceback.format_exc())
    data, last_update = sync_data(data,
                                  last_update,
                                  Constants().nightly_values_remote_file,
                                  path_to_file)
    return data, last_server_update


def get_launch_property(launch, prop_name):
    for launch_property in launch['pack']['properties']:
        if launch_property['key'] == prop_name:
            return launch_property['value']
    return None


def get_pack_environment(pack):
    if 'devices' in pack:
        return pack['devices']
    else:
        return pack['browsers']
