# -*- coding: utf-8 -*-
import threading
from abc import ABC
from datetime import datetime, timedelta, timezone
from typing import List, Dict

from autotest_run.autotest_run_functions import *
from autotest_run.run import RunType, add_run, Run, set_run_running, RunStatus
from autotest_run.run_queue import RunQueue
from packs_config import liza_packs, touch_packs, cal_packs
from utils.config import Constants
from utils.send_message import send_message
from utils.utils import Service

FORMAT = '%(asctime)s:%(levelname)s:%(name)s:%(message)s'
logging.basicConfig(level=logging.DEBUG, format=FORMAT)
log = logging.getLogger("nightlyAutotestRunner")

_test_runner_instances = {}


class TestRunner(ABC, threading.Thread):
    RUN_NOT_IN_TOMATO = "runned not from tomato"
    headers = {
        "Content-Type": "application/json"
    }
    report_emails = ['mail-auto-reports@yandex-team.ru']

    @classmethod
    def get_instance(cls):
        if cls not in _test_runner_instances:
            _test_runner_instances[cls] = cls()
        return _test_runner_instances[cls]

    @property
    def size(self):
        return self.queue.size()

    def __init__(self, service=Service.liza.value, all_packs=liza_packs.packs, max_overlap=2):
        threading.Thread.__init__(self, name=service)

        self.service = service
        self.queue = RunQueue(service)
        self.current_runs: List[Run] = []
        self.all_packs = all_packs
        self.max_overlap = max_overlap
        self.results: Dict[str, Dict[str, Dict[str, Dict[str, any]]]] = {}
        self.raw_results = {}
        self.starting_runs: Dict[str, Run] = {}
        self.sent_reports = []
        self.idle_space_for_packs: Dict[str, int] = {}
        self.pack_times = {}

    def add(self, args, priority=2, run_type=RunType.regular):
        run = add_run(args, run_type)

        self.queue.add(run, priority)

        return run

    def remove(self, run_id):
        if self.get_status(run_id) == RunStatus.waiting.name:
            self.queue.delete_run(run_id)
        else:
            self.kill_running_run(run_id)

    def get_status(self, remove_run_id):
        for run in self.get_queue_and_current():
            if run.run_id == remove_run_id:
                return run.status.name

    def kill_running_run(self, remove_run_id):
        for pack in self.results:
            if remove_run_id in self.results[pack]:
                for environment in self.results[pack][remove_run_id]:
                    launch = self.results[pack][remove_run_id][environment]
                    requests.delete(
                        'http://aqua.yandex-team.ru/aqua-api/services/launch/%s' % launch['id'],
                        headers={"Content-Type": "application/json"},
                        verify=False,
                    )
                    log.info(f'killing pack {launch["id"]} for run {remove_run_id}')

    def run(self):
        while True:
            try:
                self.update_current_results()
                self.retry_runs()
                self.run_autotests()
                self.send_finished_reports_and_run_after_run()
                self.kill_long_runs()
            except (KeyboardInterrupt, SystemExit):
                raise
            except Exception as e:
                log.warning('Unknown error')
                log.error(traceback.format_exc())
            sleep(20)

    def get_queue_and_current(self):
        return self.current_runs + list(self.starting_runs.values()) + self.queue.get_all()

    def update_current_results(self):
        self.results = {}
        self.pack_times = {}
        for pack in self.all_packs:
            overlap = self.all_packs[pack].get('max_overlap', self.max_overlap)
            self.idle_space_for_packs[pack] = overlap
            self.results[pack] = {}
            self.pack_times[pack] = {}
            pack_info = requests.get(
                f'http://aqua.yandex-team.ru/aqua-api/services/launch/page/simple?limit={overlap * (retries[self.service] + 1) * self.get_environments_num_for_pack(self.all_packs[pack]) * 2}&packId={pack}&skip=0',
                headers=self.headers,
                verify=False,
            ).json()['launches']

            for launch in pack_info:
                if (launch['launchStatus'] in running_test_statuses) and (launch.get('tag', '') != 'borador_production') and not self.is_launch_running_to_long(launch):
                    self.idle_space_for_packs[pack] -= 1

            self.raw_results[pack] = pack_info

            for launch in pack_info:
                tomato_id = get_launch_property(launch, Constants().id_param)
                tomato_retry = get_launch_property(launch, Constants().retries_param)
                tomato_retry = 0 if tomato_retry is None else int(tomato_retry)
                if tomato_id is None:
                    continue
                browser = get_launch_property(launch, 'webdriver.driver')
                device = get_launch_property(launch, 'deviceType')
                environment = device if device else browser
                if tomato_id not in self.results[pack]:
                    self.results[pack][tomato_id] = {}
                    self.pack_times[pack][tomato_id] = {}
                if tomato_retry == 1:
                    self.pack_times[pack][tomato_id][environment] = launch['startTime']
                if environment not in self.results[pack][tomato_id]:
                    self.results[pack][tomato_id][environment] = launch
                else:
                    if int(launch['createdTime']) > int(self.results[pack][tomato_id][environment]['createdTime']):
                        self.results[pack][tomato_id][environment] = launch

    def retry_runs(self):
        for pack in self.results:
            if self.idle_space_for_packs[pack] < 1:
                continue
            for tomato_id in self.results[pack]:
                for environment in self.results[pack][tomato_id]:
                    launch = self.results[pack][tomato_id][environment]
                    retries_done = int(get_launch_property(launch, Constants().retries_param))
                    if (retries_done > retries[self.service]) | (self.idle_space_for_packs[pack] < 1) | self.is_launch_outdated(launch):
                        continue
                    if (launch['failedSuites'] != 0) & (launch['launchStatus'] not in running_test_statuses) & (launch['revokedSuites'] == 0):
                        # Рестарт упавших
                        self.retry_pack(pack, launch, launch['id'], retries_done)

    def get_actual_finished_ids(self):
        ids_status = {}
        finished = []
        for pack in self.results:
            for id in self.results[pack]:
                if id in self.starting_runs:
                    continue
                if id not in ids_status:
                    ids_status[id] = {}
                if pack not in ids_status[id]:
                    ids_status[id][pack] = {}
                for environment in get_pack_environment(self.all_packs[pack]):
                    ids_status[id][pack][environment] = 'NO_INFO'
                for environment in self.results[pack][id]:
                    launch = self.results[pack][id][environment]
                    if self.is_launch_outdated(launch):
                        ids_status[id][pack][environment] = 'OUTDATED'
                        continue
                    if (launch['launchStatus'] not in finished_run_statuses) | ((int(launch['failedSuites']) != 0) & (int(get_launch_property(launch, Constants().retries_param)) <= retries[self.service])):
                        ids_status[id][pack][environment] = 'RUNNING'
                        continue
                    if (int(launch['failedSuites']) == 0) | (int(get_launch_property(launch, Constants().retries_param)) > retries[self.service]) | (int(launch['revokedSuites']) != 0):
                        ids_status[id][pack][environment] = 'FINISHED'
        for id in ids_status:
            is_finished = True
            has_outdated = False
            outdated = []
            for pack in ids_status[id]:
                for environment in ids_status[id][pack]:
                    if ids_status[id][pack][environment] != 'FINISHED':
                        if ids_status[id][pack][environment] == 'OUTDATED':
                            has_outdated = True
                            outdated.append(f'{pack} {environment}')
                        else:
                            is_finished = False
            if is_finished:
                if has_outdated:
                    run_to_remove = None
                    for run in self.current_runs:
                        if run.run_id == id:
                            run_to_remove = run
                    if run_to_remove is not None:
                        log.info(f'run with tomato id: {id} is outdated but in current runs, adding it to finished anyway')
                        log.info(f'outdated results are: {outdated}')
                        finished.append(id)
                else:
                    finished.append(id)
        if len(list(set(finished) - set(self.sent_reports))) > 0:
            log.info(f'ids info: {ids_status}')
        return list(set(finished) - set(self.sent_reports))

    def run_autotests(self):
        runs_to_remove_from_starting = []
        if self.queue.size() > 0:
            while len(self.starting_runs) < self.max_overlap * 2:
                if self.queue.size() == 0:
                    break
                next_run: Run = self.queue.get_next()['run']
                if next_run.run_type.name == RunType.nightly.name:
                    self.add_night_runs()
                else:
                    self.starting_runs[next_run.run_id] = next_run
                    log.info(f'run {next_run.run_id} added to start')
                log.info(f'starting runs total: {self.starting_runs}')
        for starting_run_id in self.starting_runs:
            starting_run = self.starting_runs[starting_run_id]
            is_all_run = True
            for pack in starting_run.args['packs']:
                if starting_run_id in self.results[pack]:
                    if len(self.results[pack][starting_run_id]) == len(get_pack_environment(self.all_packs[pack])):
                        continue
                is_all_run = False
                if self.idle_space_for_packs[pack] > 0:
                    self.run_autotests_for_service(starting_run, pack)
            if is_all_run:
                runs_to_remove_from_starting.append(starting_run_id)
        for run in runs_to_remove_from_starting:
            log.info(f'all packs for run {run} is run, removing from checking')
            self.current_runs.append(self.starting_runs.pop(run))
            log.info(f'starting runs: {self.starting_runs}')
            log.info(f'current runs: {self.current_runs}')

    def send_finished_reports_and_run_after_run(self):
        # TODO: в raw_results может не быть первого запуска, надо придумать, как хранить время старта либо дозапрашивать
        finished_ids = self.get_actual_finished_ids()
        if finished_ids:
            log.info(f'finished {set(finished_ids)} runs')
        for id in finished_ids:
            finished_run = list([run for run in self.current_runs if run.run_id == id])
            if len(finished_run) > 0:
                finished_run = finished_run[0]
            else:
                finished_run = None
            if id in self.sent_reports:
                continue
            message = ''
            host = ''
            message_error = []
            message_ok = []
            emails = self.report_emails
            for pack in self.results:
                if id not in self.results[pack]:
                    continue
                for environment in self.results[pack][id]:
                    launch = self.results[pack][id][environment]
                    if len(message) == 0:
                        host = get_launch_property(launch, 'webdriver.base.url')
                    if len(emails) == len(self.report_emails):
                        emails = emails + get_launch_property(launch, Constants().emails_param).split(',')
                    failed = launch['failedSuites']
                    passed = launch['passedSuites']
                    total = launch['totalSuites']
                    pack_name = launch['pack']['name']
                    report_url = launch.get('reportUrl', '')
                    browser = get_launch_property(launch, 'webdriver.driver')
                    device = get_launch_property(launch, 'deviceType')
                    if environment in self.pack_times[pack][id]:
                        execution_time = (int(launch['stopTime']) - self.pack_times[pack][id][environment]) / 1000.0 / 60.0
                        execution_time = round(execution_time, 1)
                    else:
                        execution_time = 0
                    environment = browser if device is None else device
                    if failed != 0:
                        log.info(f'Failed pack {id}.')
                        message_error.append(f'Проблемы в паке "{pack_name}" в {environment}. Упало классов: {failed}. Время: {execution_time} минут. Репорт: {report_url}.')
                    elif passed != total:
                        log.info(f'Problem pack {id}. {launch}')
                        message_error.append(f'Не смог понять успешность прогона пака "{pack_name}" в {environment}. Время: {execution_time} минут. Репорт: {report_url}')
                    else:
                        log.info(f'Successful pack {id}. {launch}')
                        message_ok.append(f'Успешно прошёл пак "{pack_name}" в {environment}. Время: {execution_time} минут. Репорт: {report_url}')
            log.info(f'run {id} is finished, sending message')
            message = self.convert_message_to_html('\n'.join([f'Хост {host}'] + message_error + message_ok))
            send_message(emails, 'Отчёт по автотестам', message)
            self.sent_reports.append(id)
            # TODO: придумать как не завязываться на current_runs или научиться собирать current_runs из паков аквы
            if finished_run is not None:
                log.info(f'message for run {id} is sent, do after run and remove run from current')
                self.current_runs.remove(finished_run)
                if finished_run.args['ticket'] != '':
                    send_startreck_comment(finished_run.args['ticket'], message)
                self.after_run(finished_run)

    def run_autotests_for_service(self, run, pack_id):
        log.info(f'preparing properties for pack {pack_id} for run {run.run_id}')
        pack = self.all_packs[pack_id]
        browser_versions = {
            'chrome': '91.0',
            'firefox': '76.0',
            'internet explorer': '11',
            'yandex-browser': '18.10.0.444'
        }
        run_parameters = {
            Constants().retries_param: '1',
            Constants().id_param: run.run_id,
            Constants().emails_param: ','.join(run.args['emails']),
            'hazelcast.semaphore.permits': '80'
        }
        if 'corp' in pack.get('category', []):
            host = run.args['corp_stand']
        else:
            host = run.args['stand']
            if (run.args.get('category', '') == 'smoke') & ('mail' in host):
                run_parameters['webdriver.prod.url'] = 'https://trunk.mail.yandex.ru'
        run_parameters['webdriver.base.url'] = host if 'https' in host else 'https://%s' % host
        pack = self.all_packs[pack_id]
        environments = get_pack_environment(pack)
        env_number = -1
        for environment in environments:
            env_number += 1
            version = ''
            if 'versions' in pack:
                version = pack['versions'][env_number]
            elif environment in browser_versions:
                version = browser_versions[environment]

            if run.run_id in self.results[pack_id]:
                if environment in self.results[pack_id][run.run_id]:
                    continue
            if self.idle_space_for_packs[pack_id] < 1:
                break
            if environment in browser_versions:
                run_parameters['webdriver.driver'] = environment
                if version:
                    run_parameters['webdriver.version'] = version
            else:
                run_parameters['webdriver.driver'] = 'chrome'
                run_parameters['deviceType'] = environment
            if 'semi-corp' in pack.get('category', []):
                if 'https' in run.args['corp_stand']:
                    run_parameters['webdriver.corp.url'] = run.args['corp_stand']
                else:
                    run_parameters['webdriver.corp.url'] = f'https://{run.args["corp_stand"]}'
            if 'additional_properties' in pack:
                run_parameters = dict(**run_parameters, **pack['additional_properties'])

            pack_info = requests.get(
                'http://aqua.yandex-team.ru/aqua-api/services/pack/%s' % pack_id,
                headers=self.headers,
                verify=False,
            ).json()

            data_template = pack_info

            for prop in run_parameters:
                is_changed = 0
                for pack_prop in data_template['properties']:
                    if pack_prop['key'] == prop:
                        pack_prop['value'] = run_parameters[prop]
                        is_changed = 1
                        break
                if not is_changed:
                    data_template['properties'].append({'key': prop, 'value': run_parameters[prop]})
            log.info(f'launching pack {pack_id} for run {run.run_id} in environment {environment}')
            self.idle_space_for_packs[pack_id] -= 1
            r = requests.put(
                'http://aqua.yandex-team.ru/aqua-api/services/launch/pack',
                headers=self.headers,
                verify=False,
                json=data_template
            )
            launch_id = r.json()['id']
            log.info(f'launched pack {pack_id} for run {run.run_id}, launch id is {launch_id}')
            set_run_running(run.run_id)

    @staticmethod
    def convert_message_to_html(message: str):
        return message.replace('\n', '<br>') \
            .replace('Проблемы', '<font color="red">Проблемы</font>') \
            .replace('Успешно', '<font color="green">Успешно</font>') \
            .replace('Не смог понять успешность', '<font color="brown">Не смог понять успешность</font>')

    def add_night_runs(self):
        data = get_runs_data(self.service)
        for task in data:
            emails = get_emails(task, self.service)
            run_args = {
                'category': [''],
                'service': self.service,
                'stand': task['stand'],
                'corp_stand': task['corp_stand'],
                'emails': emails, 'ticket': '',
                'packs': get_packs_by_category(self.all_packs, [''])
            }
            self.add(run_args)

    def kill_long_runs(self):
        msk_timezone = timezone(timedelta(hours=3))
        for pack in self.results:
            for launch_id in self.results[pack]:
                for environment in self.results[pack][launch_id]:
                    launch = self.results[pack][launch_id][environment]
                    retries_done = int(get_launch_property(launch, Constants().retries_param))
                    if launch['launchStatus'] in running_test_statuses:
                        if self.is_launch_running_to_long(launch):
                            log.warning("Test with id %s is running for too long! Breaking" % launch['id'])
                            requests.delete(
                                'http://aqua.yandex-team.ru/aqua-api/services/launch/%s' % launch['id'],
                                headers=self.headers,
                                verify=False,
                            )
                            self.retry_pack(pack, launch, launch['id'], retries_done)

    def retry_pack(self, pack, launch, launch_id, retries_done):
        log.info(f'restarting pack {pack} for run {get_launch_property(launch, Constants().id_param)}')
        requests.get(
            f'http://aqua.yandex-team.ru/aqua-api/services/launch/{launch_id}/restart?failed-only=true&{Constants().retries_param}={retries_done + 1}',
            headers=self.headers,
            verify=False,
        )
        self.idle_space_for_packs[pack] -= 1

    def after_run(self, run: Run):
        pass

    @staticmethod
    def is_launch_running_to_long(launch):
        msk_timezone = timezone(timedelta(hours=3))
        if datetime.now(msk_timezone) - datetime.fromtimestamp(int(launch['startTime']) / 1e3, tz=msk_timezone) > timedelta(hours=2):
            return True
        return False

    @staticmethod
    def is_launch_outdated(launch):
        msk_timezone = timezone(timedelta(hours=3))
        now = datetime.now(msk_timezone)
        launch_start = datetime.fromtimestamp(int(launch['stopTime']) / 1e3, tz=msk_timezone)
        if now - launch_start > timedelta(hours=4):
            log.debug(f'outdated launch {launch["id"]}. Now is {now.timestamp()}, launch started at {int(launch["stopTime"]) / 1e3}')
            return int(launch['stopTime']) != 0
        return False

    @staticmethod
    def get_environments_num_for_pack(pack):
        return max(len(pack.get('browsers', [])), len(pack.get('devices', [])))


class AutoTestRunner(TestRunner):

    def __init__(self):
        TestRunner.__init__(self, Service.liza.value, liza_packs.packs)

    def after_run(self, run):
        args = run.args

        # Запускаем греп отдельным тредом, чтобы не мешался.
        if (args.get('category', '') != 'smoke') & (run.run_type.name != RunType.nightly.name):
            if re.fullmatch(r'(http[s]?://)?mail.yandex.ru[/]?', args['stand']):
                log.info(f'run {run.run_id} is for prod, skipping grep')
            else:
                log.info(f'start grep for {run.run_id} and stand {args["stand"]}')
                grep_thread = threading.Thread(target=send_message_with_booster_urls,
                                               args=(args['stand'], args['service'], args['emails'], args.get('ticket', '')))
                grep_thread.start()
        super().after_run(run)


class TouchTestsRunner(TestRunner):
    def __init__(self):
        TestRunner.__init__(self, Service.touch.value, touch_packs.packs, max_overlap=2)


class CalTestsRunner(TestRunner):
    def __init__(self):
        TestRunner.__init__(self, Service.cal.value, cal_packs.packs)


def wait_till_all_packs_finished(packs):
    print("waiting for packs to finish...")
    headers = {"Content-Type": "application/json"}
    if len(packs) == 0:
        return True
    wait_start_time = time()
    while time() - wait_start_time < 1.5 * 60 * 60:
        is_finished = True
        for pack in packs:
            r = requests.get(
                'http://aqua.yandex-team.ru/aqua-api/services/launch/page/simple?limit=3&packId=%s&skip=0' % pack,
                headers=headers,
                verify=False)
            if r.json()['launches'][0]['launchStatus'] in running_test_statuses:
                if r.json()['launches'][0].get('tag', '') != 'borador_production':
                    is_finished = False
        if is_finished:
            return True
        log.info('Автотесты заняты, жду')
        sleep(20)
