from enum import Enum
from threading import Lock, Thread
from time import sleep, time
from uuid import uuid4

current_runs = {}
lock = Lock()

CLEANER_INTERVAL = 60 * 60
OLD_RUN_AGE = 3 * 60 * 60


class RunStatus(Enum):
    waiting = 1
    running = 2
    complete = 3


class RunType(Enum):
    regular = 1
    nightly = 2


class Run(object):
    run_id = None
    run_type = RunType.regular
    args = {}
    status = RunStatus.waiting
    launches = []
    created_at = 0
    time_to_run = 0

    ESTIMATE_TIME_SMOKE_LIZA = 20
    ESTIMATE_TIME_REGULAR_LIZA = 60
    ESTIMATE_TIME_NIGHTLY_LIZA = 40

    def __init__(self, args, run_type=RunType.regular):
        self.run_id = str(uuid4())
        self.run_type = run_type
        self.args = args
        self.created_at = time()

    def to_short_json(self):
        return {"run_id": self.run_id, "status": self.get_status_string(), "launches": self.launches}

    def to_json_runned(self):
        return {
            "email": self.get_login_string(),
            'runtype': self.get_runtype_string(),
            'estimate': self.time_to_run,
            'run_id': self.run_id
        }

    def get_status_string(self):
        if self.status == RunStatus.waiting:
            return "waiting"
        elif self.status == RunStatus.running:
            return "running"
        else:
            return "complete"

    def get_login_string(self):
        if 'emails' in self.args:
            email = self.args['emails']
            return str(email).replace("@yandex-team.ru", "").replace("['", "").replace("']", "").replace("'", "")
        return ''

    def get_runtype_string(self):
        if self.args.get('category') == "smoke":
            return "smoke"
        return self.run_type.name

    def estimate_time(self):
        if self.get_runtype_string() == "smoke":
            return self.ESTIMATE_TIME_SMOKE_LIZA
        if self.get_runtype_string() == "regular":
            return self.ESTIMATE_TIME_REGULAR_LIZA
        if self.get_runtype_string() == "nightly":
            return self.ESTIMATE_TIME_NIGHTLY_LIZA

    def to_json(self):
        return {
            'run_id': self.run_id,
            'run_type': self.run_type.value,
            'args': self.args,
            'status': self.status.value,
            'launches': self.launches,
            'created_at': self.created_at,
            'time_to_run': self.time_to_run
        }

    def from_json(self, run_json):
        self.run_id = run_json['run_id']
        self.run_type = RunType(int(run_json['run_type']))
        self.args = run_json['args']
        self.status = RunStatus(int(run_json['status']))
        self.launches = run_json['launches']
        self.created_at = run_json['created_at']
        self.time_to_run = run_json['time_to_run']
        return self

    def __str__(self):
        return f"{self.to_json()}"


class RunsCleaner(Thread):
    def run(self):
        while True:
            with lock:
                ids_to_clean = []
                now = time()

                for (run_id, run) in current_runs.items():
                    if now - run.created_at > OLD_RUN_AGE:
                        ids_to_clean.append(run_id)

                for run_id in ids_to_clean:
                    del current_runs[run_id]

            sleep(CLEANER_INTERVAL)


def add_run(args, run_type=RunType.regular):
    run = Run(args, run_type)

    current_runs[run.run_id] = run

    return run


def add_launch(run_id, pack, report, launch, browser_or_device, time):
    with lock:
        if run_id in current_runs:
            run = current_runs[run_id]
            run.launches.append({
                "pack": pack,
                "report": report,
                "launch": launch,
                "browser_or_device": browser_or_device,
                "time": time
            })


def set_run_status(run_id, status):
    with lock:
        if run_id in current_runs:
            run = current_runs[run_id]
            run.status = status


def set_run_running(run_id):
    set_run_status(run_id, RunStatus.running)


def set_run_complete(run_id):
    set_run_status(run_id, RunStatus.complete)


def get_run(run_id):
    return current_runs.get(run_id)
