import asyncio
import codecs
import json
import logging
import os
import re
import sys
import time
from pathlib import Path

import requests
import tornado.escape
import tornado.httpserver
import tornado.ioloop
import tornado.options
import tornado.web
from tornado.options import define, options

sys.path.append(str(Path(__file__).parent.parent))
dir_path = os.path.dirname(os.path.realpath(__file__))
sys.path.insert(0, dir_path)

from assessors.api_classes.testpalm import RunCreationMode
from utils.utils import Service

from assessors_run.assessor_run_status import AssessorRunner
from assessors_run.assessors_queue import AssessorsRun

from autotest_run.autotest_run_functions import get_sync_nightly_values
from utils.config import Constants
from assessors_run.assessors_queue import BookingMode
from assessors.api_classes.booking import get_closest_free_place
from autotest_run.autotest_run_functions import running_test_statuses, get_packs_by_category
from packs_config.liza_packs import packs as packs_liza
from packs_config.cal_packs import packs as packs_cal
from packs_config.touch_packs import packs as packs_quinn
from autotest_run.queue_run import AutoTestRunner, TouchTestsRunner, CalTestsRunner, TestRunner
from autotest_run.run import get_run, RunType, RunsCleaner, RunStatus
from Create_tickets_and_version.StaffStandart import maintv
from set_secret import set_secret

FORMAT = '%(asctime)s:%(levelname)s:%(name)s:%(message)s'
logging.basicConfig(level=logging.INFO, format=FORMAT)
log = logging.getLogger("nightlyAutotestRunner")
scriptPath = os.path.dirname(os.path.abspath(__file__))
define("port", default=80, help="run on the given port", type=int)

stand_list = [
    'mail.yandex.ru',
    'ub1-qa.mail.yandex.ru',
    'ub2-qa.mail.yandex.ru',
    'ub3-qa.mail.yandex.ru',
    'ub4-qa.mail.yandex.ru',
    'ub5-qa.mail.yandex.ru',
    'ub6-qa.mail.yandex.ru',
    'ub7-qa.mail.yandex.ru',
    'ub8-qa.mail.yandex.ru',
    'ub9-qa.mail.yandex.ru',
    'ub10-qa.mail.yandex.ru',
    'ub11-qa.mail.yandex.ru',
    'ub12-qa.mail.yandex.ru',
]


def get_actual_versions():
    start = time.time()
    url_liza = "https://testpalm-api.yandex-team.ru:443/version/mail-liza/?archived=false"
    url_touch = "https://testpalm-api.yandex-team.ru:443/version/mail-touch/?archived=false"
    url_cal = "https://testpalm-api.yandex-team.ru:443/version/cal/?archived=false"
    req_headers = {'Authorization': f'OAuth {os.environ["TESTPALM_OAUTH"]}', 'Content-Type': 'application/json'}
    response_liza = requests.get(url_liza, headers=req_headers, verify=False)
    response_touch = requests.get(url_touch, headers=req_headers, verify=False)
    response_cal = requests.get(url_cal, headers=req_headers, verify=False)
    response_touch = [] if response_touch.status_code != 200 else response_touch.json()
    response_liza = [] if response_liza.status_code != 200 else response_liza.json()
    response_cal = [] if response_cal.status_code != 200 else response_cal.json()
    all_versions = [version["title"] for version in response_liza + response_touch + response_cal]
    logging.info(f'Получил версиии из пальмы за {time.time() - start}')
    return all_versions


class MainHandler(tornado.web.RequestHandler):
    path_to_file = Constants().nightly_values_local_file

    def get(self):
        start = time.time()
        print(self.path_to_file)
        with open(self.path_to_file, 'r') as f:
            try:
                saved_tasks = json.load(f)
                for task in saved_tasks:
                    if 'service' not in task:
                        task['service'] = Service.liza.value
                    if 'versions' not in task:
                        task['versions'] = ''
            except ValueError:
                print('no json found, setting empty')
                saved_tasks = {}
        logging.info(f'Переходим к рендеру страницы за {time.time() - start} от начала')
        self.render('pages/web.html', saved_tasks=saved_tasks, stand_list=stand_list, tp_versions=get_actual_versions())

    def post(self):
        print(self.request.body_arguments)
        print(self.request.query_arguments)
        post_query_args = self.request.query_arguments
        if post_query_args['action'] == [b'add']:
            self.add_task()
        elif post_query_args['action'] == [b'edit']:
            self.edit_task()
        elif post_query_args['action'] == [b'assessor']:
            self.run_assessors()
        elif post_query_args['action'] == [b'assessor-filter']:
            self.run_assessors_filter()
        elif post_query_args['action'] == [b'run_now']:
            self.run_now()
        elif post_query_args['action'] == [b'remove_run']:
            self.remove_run()
        elif post_query_args['action'] == [b'create-version']:
            self.create_version_mobmail()
        else:
            log.error('I dont know what to do')

    def remove_run(self):
        post_args = self.request.arguments
        if all(k in post_args for k in ('run_id', 'service')):
            remove_run_id = self.get_argument('run_id')
            service = self.get_argument('service')
            if service == Service.liza.value:
                AutoTestRunner.get_instance().remove(remove_run_id)
            elif service == Service.touch.value:
                TouchTestsRunner.get_instance().remove(remove_run_id)
            elif service == Service.cal.value:
                CalTestsRunner.get_instance().remove(remove_run_id)
        self.get()

    def add_task(self):
        emails = []
        post_args = self.request.body_arguments
        if 'stand' in post_args:
            stand = self.get_argument('stand')
            corp_stand = self.get_argument('corp_stand')
            service = self.get_argument('service', default=Service.liza.value)
            stand_email = [s.strip() for s in re.split(r'[ ]*[,;][ ]*', self.get_arguments('emails')[0])]
            if stand_email[0] != '':
                emails = stand_email
            stands_json = {'stand': stand, 'corp_stand': corp_stand, 'emails': emails, 'service': service}
            # get_sync_nightly_values()
            with codecs.open(self.path_to_file, 'r', encoding='utf-8') as f:
                prev_tasks = json.load(f)
            for task in prev_tasks:
                if task['stand'] == stand and task['service'] == service:
                    prev_tasks.remove(task)
            prev_tasks.append(stands_json)
            with codecs.open(self.path_to_file, 'w', encoding='utf-8') as f:
                json.dump(prev_tasks, f, ensure_ascii=False)
            # get_sync_nightly_values()
            log.info(stands_json)
            self.render('pages/results.html', stands=stand)

    def run_now(self):
        start = time.time()
        if self.get_argument('action') == 'just_to_ask':
            log.info('just_to_ask')
            priority = 1
        else:
            log.info('run_now')
            priority = 2
        stand = self.get_argument('stand')
        corp_stand = self.get_argument('corp_stand')
        service = self.get_argument('service', default=Service.liza.value)
        # К нам ходят по апи, поэтому тут сделал конвертацию параметра touch в service
        touch_param = self.get_argument('touch', default='')
        if touch_param == 'on':
            service = Service.touch.value
        elif touch_param == 'off':
            service = Service.liza.value
        category = self.get_argument('category', default=[''])
        stand_email = [s.strip() for s in re.split(r'[ ]*[,;][ ]*', self.get_arguments('emails')[0])]
        if stand_email[0] != '':
            emails = stand_email
        else:
            emails = []
        logging.info(f'Попарсил все аргументы за {time.time() - start} от начала')
        run_tests_with_params(stand, corp_stand, service, category, emails, priority=priority)
        logging.info(f'Начинаю рендер за {time.time() - start} от начала')
        self.render('pages/results.html', stands=stand)

    def edit_task(self):
        """
        Функция изменения или удаления задачи.
        В body_arguments аргументы приходят в формате откуда_параметр_hostзадачи_touchзадачи,
        например modal_host_ub1-qa.yandex.ru_on значает, что это парметр host из
        формы в модальном окне, для задачи на стенде ub1-qa.yandex.ru в таче.
        Функция собирает параметры в слоаварь по ключам, каждый ключ - отдельная задача. В данном случае ключ -
        комбинация их параметров host и touch так как предполагаем, что не может быть двух задач с одинаковыми
        host и touch. Для этого, мы разбиваем аргумент в список :arg: на четыре части, описанных выше, тогда
        arg[2]+arg[3] - уникальный ключ задачи в словаре
        arg[1] - параметр задачи (host/touch/emails/versions/remove)
        Если в аргументах для уникального ключа есть remove:on (активирован чекбокс удаления), то такую задачу мы не
        записываем.
        После сбора словаря, собираем чистовой список новых задач и записываем их в файл
        """
        post_args = self.request.body_arguments
        tasks_draft = {}
        tasks = []
        for arg_raw in post_args:
            arg = arg_raw.split('__')
            if arg[2] + arg[3] not in tasks_draft:
                tasks_draft[arg[2] + arg[3]] = {}
            if arg[1] == 'emails':
                tasks_draft[arg[2] + arg[3]][arg[1]] = re.split(r'[ ]*[,;][ ]*', post_args[arg_raw][0].decode('utf-8'))
            else:
                tasks_draft[arg[2] + arg[3]][arg[1]] = post_args[arg_raw][0].decode('utf-8')
        for key in tasks_draft:
            tasks.append(tasks_draft[key])
        log.info(tasks)
        tasks_to_add = []
        for task in tasks:
            if 'service' not in task:
                task['service'] = Service.liza.value
            if 'remove' not in task:
                tasks_to_add.append(task)
        with codecs.open(self.path_to_file, 'w', encoding='utf-8') as f:
            json.dump(tasks_to_add, f, ensure_ascii=False)
        self.redirect('/')

    def run_assessors(self):
        args = self.request.body_arguments
        post_args = {d: tornado.escape.to_unicode(args[d][0]) for d in args}
        run_earlier = post_args.get('runEarlier', '')
        booking_id = post_args.get('bookingId', '')
        log.info(post_args)
        AssessorRunner.get_instance().add(
            AssessorsRun(
                post_args['testpalm'],
                post_args['service'],
                post_args['spec_conditions'],
                post_args['stand'],
                post_args['requester'],
                BookingMode(post_args['bookingMode']),
                version_config_name=post_args['version_chooser'],
                run_earlier=run_earlier,
                booking_id=booking_id
            )
        )
        self.render('pages/assessors_run.html', testpalm_version=post_args['testpalm'])

    def run_assessors_filter(self):
        args = self.request.body_arguments
        log.info(args)
        environment = [tornado.escape.to_unicode(x) for x in args['environment']]
        post_args = {d: tornado.escape.to_unicode(args[d][0]) for d in args}
        log.info((post_args['stand'], post_args['testpalm'], post_args['requester'], post_args['service'],
                 post_args['filter'], post_args['mode'], post_args['special_conditions']))
        log.info(environment)
        AssessorRunner.get_instance().add(
            AssessorsRun(
                post_args['testpalm'],
                post_args['service'],
                post_args['special_conditions'],
                post_args['stand'],
                post_args['requester'],
                BookingMode.new,
                version_config_name='custom',
                run_earlier=True,
                mode=RunCreationMode(int(post_args['mode'])),
                filter_tp=post_args['filter'],
                environment=environment
            )
        )
        self.render('pages/assessors_run.html', testpalm_version=post_args['testpalm'])

    def create_version_mobmail(self):
        args = self.request.body_arguments
        log.info(args)
        post_args = {d: tornado.escape.to_unicode(args[d][0]) for d in args}
        log.info((post_args['requester_mm'], post_args['mobile_platform'], post_args['mobileapp_version'], post_args['mobile_dashboard']))
        mainttv = maintv(
            str.strip(post_args['requester_mm']),
            str.strip(post_args['mobile_platform']),
            str.strip(post_args['mobileapp_version']),
            str.strip(post_args['mobile_dashboard']),
            post_args['release_ticket_testpalm'],
            post_args.get('release_notes', ''),
            post_args.get('update_libs', ''),
            post_args.get('android_beta', '')
        )
        self.render('pages/version_created.html', mobile_platform=post_args['mobile_platform'])


class ApiHandler(tornado.web.RequestHandler):
    def get(self):
        runner: TestRunner
        packs = ''
        post_query_args = self.request.query_arguments
        if post_query_args.get('project', ['err'])[0] == b'liza':
            packs = packs_liza
            runner = AutoTestRunner.get_instance()
        elif post_query_args.get('project', ['err'])[0] == b'touch':
            packs = packs_quinn
            runner = TouchTestsRunner.get_instance()
        elif post_query_args.get('project', ['err'])[0] == b'cal':
            packs = packs_cal
            runner = CalTestsRunner.get_instance()
        else:
            print(post_query_args.get('project', ''))
            self.write(json.dumps({'error': 'no such project'}))
            return

        # if runner.current_runs:
        #     runner.current_runs.time_to_run = '???'

        time_to_run = 0
        for run in runner.queue.get_all():
            run.time_to_run = run.estimate_time() + time_to_run
            time_to_run = run.time_to_run

        if (not runner.current_runs) or (runner.current_runs[0].status.name == RunStatus.waiting.name):
            headers = {"Content-Type": "application/json"}
            for pack in packs:
                r = requests.get(
                    'http://aqua.yandex-team.ru/aqua-api/services/launch/page/simple?limit=3&packId=%s&skip=0' % pack,
                    headers=headers,
                    verify=False)
                if len(r.json()['launches']) == 0:
                    continue
                if r.json()['launches'][0]['launchStatus'] in running_test_statuses:
                    self.write(
                        json.dumps({
                            'run': [{"email": r.json()['launches'][0]['user'], 'runtype': 'internal aqua run',
                                     'estimate': '', 'run_id': ''}] +
                                   [run.to_json_runned() for run in runner.get_queue_and_current()]
                                   })
                    )
                    return

        self.write(json.dumps({'run': [run.to_json_runned() for run in runner.get_queue_and_current()]}))

    def post(self):
        self.write({})


class ApiStatusHandler(tornado.web.RequestHandler):
    def get(self):
        run_id = self.get_query_argument('run_id', '')
        print(run_id)
        run = get_run(run_id)
        if run is not None:
            self.write(run.to_short_json())
        else:
            self.set_status(404)
            self.write({})


class ApiRunHandler(tornado.web.RequestHandler):
    def get(self):
        self.write(json.dumps({'error': 'use POST method'}))

    def post(self):
        data = json.loads(self.request.body)
        log.info(data)
        stand = data.get('stand')
        corp_stand = data.get('corp_stand', '')
        service = data.get('service', Service.liza.value)
        # К нам ходят по апи, поэтому тут сделал конвертацию параметра touch в service
        touch_param = data.get('touch', '')
        if touch_param == 'on':
            service = Service.touch.value
        elif touch_param == 'off':
            service = Service.liza.value
        log.info(service)
        category = data.get('category', [''])
        stand_email = [s.strip() for s in re.split(r'[ ]*[,;][ ]*', data.get('emails', ''))]
        if stand_email[0] != '':
            emails = stand_email
        else:
            emails = []
        ticket = data.get('issueKey', '')
        priority = int(data.get('priority', 2))
        run = run_tests_with_params(stand, corp_stand, service, category, emails, ticket=ticket, priority=priority)
        self.write(json.dumps({'status': 'ok', 'id': run.run_id}))


class ApiNightlyHandler(tornado.web.RequestHandler):
    def post(self):
        service = self.get_query_argument('service', Service.liza.value)
        if service == Service.touch.value:
            run = TouchTestsRunner.get_instance().add({'packs': packs_quinn, 'service': Service.touch.value}, 1, RunType.nightly)
            self.write(json.dumps({'status': 'ok', 'id': run.run_id}))
        elif service == Service.cal.value:
            run = CalTestsRunner.get_instance().add({'packs': packs_cal, 'service': Service.cal.value}, 1, RunType.nightly)
            self.write(json.dumps({'status': 'ok', 'id': run.run_id}))
        else:
            run = AutoTestRunner.get_instance().add({'packs': packs_liza, 'service': Service.liza.value}, 1, RunType.nightly)
            self.write(json.dumps({'status': 'ok', 'id': run.run_id}))


class ApiEstimateAssessorsHandler(tornado.web.RequestHandler):
    def get(self):
        hours_to_estimate = self.get_query_argument('hours', '9')
        self.write(
            json.dumps({'closest_time': get_closest_free_place(int(hours_to_estimate)).strftime('%d.%m %H:%M:%S')})
        )


class ApiGetAssessorsRunHandler(tornado.web.RequestHandler):
    def get(self):
        self.write(
            json.dumps({
                'status': AssessorRunner.get_instance().current_run.status.name,
                'is_ok': True
            })
        )


class ApiRunAssessorsHandler(tornado.web.RequestHandler):
    def post(self):
        data = json.loads(self.request.body)
        log.info(data)
        testpalm = data['testpalm']
        service = data['service']
        special_conditions = data['special_conditions']
        stand = data['stand']
        requester = data['requester']
        version_config_name = data['version_config_name']
        mode = int(data.get('mode', 3))
        filter_tp = data.get('filter', '')
        environment = data.get('environment', [])
        run_earlier = data.get('run_earlier', '')
        booking_mode = data.get('booking_mode', '')
        booking_id = data.get('booking_id', '')

        if (version_config_name == 'custom') & (not filter_tp):
            self.write(
                json.dumps({
                    'status': 'error',
                    'error': 'custom mode can be runned only with filter'
                })
            )
            return
        if (run_earlier != '') & (booking_mode == BookingMode.regular.value):
            self.write(
                json.dumps({
                    'status': 'error',
                    'error': 'regular booking can\'t be runned with earlier flag'
                })
            )
            return
        if (booking_id == '') & (booking_mode == BookingMode.custom.value):
            self.write(
                json.dumps({
                    'status': 'error',
                    'error': 'custom booking mode needs booking id'
                })
            )
            return
        if booking_mode == '':
            self.write(
                json.dumps({
                    'status': 'error',
                    'error': 'need booking_mode'
                })
            )
            return
        AssessorRunner.get_instance().add(
            AssessorsRun(
                testpalm,
                service,
                special_conditions,
                stand,
                requester,
                BookingMode(booking_mode),
                version_config_name=version_config_name,
                mode=RunCreationMode(mode),
                filter_tp=filter_tp,
                environment=environment,
                run_earlier=run_earlier,
                booking_id=booking_id
            )
        )


def run_tests_with_params(stand, corp_stand, service, category, emails, ticket="", priority=2):
    run_args = {
        'category': category,
        'service': service,
        'stand': stand,
        'corp_stand': corp_stand,
        'emails': emails,
        'ticket': ticket
    }
    print(run_args)
    print(service)
    if Service.touch.value in service:
        print('it is touch')
        run_args['packs'] = get_packs_by_category(packs_quinn, category)
        return TouchTestsRunner.get_instance().add(run_args, priority)
    elif Service.cal.value in service:
        print('it is cal')
        run_args['packs'] = get_packs_by_category(packs_cal, category)
        return CalTestsRunner.get_instance().add(run_args, priority)
    else:
        print('it is liza')
        run_args['packs'] = get_packs_by_category(packs_liza, category)
        run_args['devices_touch'] = ''
        return AutoTestRunner.get_instance().add(run_args)


class ApiTestpalmVersionsHandler(tornado.web.RequestHandler):
    def get(self):
        self.write(
            json.dumps({'versions': get_actual_versions()})
        )


def _start_server_sync():
    while True:
        time.sleep(60)
        get_sync_nightly_values()


def main():
    if sys.platform == 'win32':
        asyncio.set_event_loop_policy(asyncio.WindowsSelectorEventLoopPolicy())

    # threading.Thread(target=_start_server_sync).start()
    print(os.path.join(scriptPath, "static"))

    set_secret.set_secrets()

    liza_test_runner = AutoTestRunner.get_instance()
    liza_test_runner.start()

    touch_test_runner = TouchTestsRunner.get_instance()
    touch_test_runner.start()

    cal_test_runner = CalTestsRunner.get_instance()
    cal_test_runner.start()

    cleaner = RunsCleaner()
    cleaner.start()

    assessor_runner = AssessorRunner.get_instance()
    assessor_runner.start()

    settings = {
        "static_path": os.path.join(scriptPath, "static"),
    }
    handlers = [
        (r"/", MainHandler),
        (r'/static/(.*)', tornado.web.StaticFileHandler, {
            'path': os.path.join(scriptPath, "static")
        }),
        (r"/api/runned", ApiHandler),
        (r"/api/run_now", ApiRunHandler),
        (r"/api/nightly", ApiNightlyHandler),
        (r"/api/status", ApiStatusHandler),
        (r"/api/estimate_assessors", ApiEstimateAssessorsHandler),
        (r"/api/assessors_status", ApiGetAssessorsRunHandler),
        (r"/api/run_assessors", ApiRunAssessorsHandler),
        (r"/api/testpalm_versions", ApiTestpalmVersionsHandler),
        (r"/(favicon.ico)", tornado.web.StaticFileHandler, {"path": ""})
    ]

    tornado.options.parse_command_line()
    application = tornado.web.Application(handlers, **settings)
    http_server = tornado.httpserver.HTTPServer(application)
    http_server.listen(options.port)
    tornado.ioloop.IOLoop.instance().start()


if __name__ == "__main__":
    main()
