import io
import json
import logging
import os
import time
import traceback

import paramiko

FORMAT = '%(asctime)s:%(levelname)s:%(name)s:%(message)s'
logging.basicConfig(level=logging.INFO, format=FORMAT)
log = logging.getLogger("sync_data")


def sync_data(data, data_last_update_timestamp, data_server_file, data_local_file):
    """
    Функция позволяет сохранять данные длительного хранения локально и на сервере. Синхронизация с сервером происходит
    раз в две минуты, синхронизация с локальным файлом происходит каждый раз.
    Все данные храняться в файлах в формате `{"data": [], "last_update": int, "last_server_update": int}`. Если файл не
    соответствует формату, то он перезатрётся.
    :param data: текущие данные, которые хранятся в переменной и которые нужно посинкать
    :param data_last_update_timestamp:  таймстемп, когда последний раз происходила синхронизация
    :param data_server_file: файл на сервере, где хранятся синхронихируемые данные
    :param data_local_file: локальный файл, где хранятся синхронизируемые данные
    :return: если данные в файле новее тех, что сейчас в переменной, то функция возвращает данные из файла и не
    перезаписывает его, если же данные в файле старее, то значения переменной записываются в файл и возвращаются
    неизменными
    """
    log.info(f'Синкаем данные для {data_local_file}')
    with open(data_local_file, 'r') as f:
        local_file_data = json.load(f)
    synced_data = data
    synced_data_last_update_timestamp = data_last_update_timestamp
    if time.time() - local_file_data['last_server_update'] >= 600:
        log.info('Давно не синкались с сервером, посинкаем сейчас')
        start = time.time()
        host = 'qa-mail-work.sas.yp-c.yandex.net'
        ssh = paramiko.SSHClient()
        keyfile = io.StringIO(os.environ["SSH_KEY"])
        key = paramiko.RSAKey.from_private_key(keyfile, password=os.environ["SSH_PWD"])
        ssh.set_missing_host_key_policy(paramiko.AutoAddPolicy())
        try:
            try:
                ssh.connect(host, pkey=key)
                sftp = ssh.open_sftp()
                sftp.get(data_server_file, data_local_file)
                sftp.close()
                ssh.close()
                log.info(f'Файл {data_server_file} подновлен с сервера')
            except Exception as e:
                log.error(f'Не удалось скачать файл {data_server_file} с сервера, возьмем последнюю локальную копию')
                log.error(traceback.format_exc())
            with open(data_local_file) as f:
                downloaded_queue = json.load(f)
                print(downloaded_queue['data'])
                log.info('Загрузили файл')
            if int(downloaded_queue['last_update']) >= data_last_update_timestamp:
                log.info('Данные новее локальных, обновляем локальные данные')
                synced_data = downloaded_queue['data']
                synced_data_last_update_timestamp = time.time()
            else:
                log.info('Данные старее локальных, обновляем серверные данные')
                with open(data_local_file, 'w') as f:
                    json.dump({'data': data, 'last_update': time.time(),
                               'last_server_update': time.time()}, f, ensure_ascii=False)
                    log.info('Данные сохранены в файл')
                ssh.connect(host, pkey=key)
                sftp = ssh.open_sftp()
                sftp.put(data_local_file, data_server_file)
                sftp.close()
                log.info('Данные залиты на сервер')
            downloaded_queue['last_server_update'] = time.time()
            with open(data_local_file, 'w') as f:
                json.dump(downloaded_queue, f, ensure_ascii=False)
            log.info(f'Посинкано за {time.time() - start} секунд')
        except Exception as e:
            log.error('Не смогли скачать данные')
            log.error(traceback.format_exc())
    else:
        log.info('Недавно синкались с сервером, так что просто посинкаемся с локальным файлом')
        if int(local_file_data['last_update']) <= data_last_update_timestamp:
            with open(data_local_file, 'w') as f:
                log.info('Данные в переменных свежее, чем в файле, сохраняем их в файл')
                json.dump({'data': data,
                           'last_update': time.time(),
                           'last_server_update': local_file_data['last_server_update']},
                          f,
                          ensure_ascii=False)
        else:
            log.info('Данные в файле свежее, чем в переменных, загружаем данные из файла')
            synced_data = local_file_data['data']
            synced_data_last_update_timestamp = time.time()
    log.info(f'Новые данные: {synced_data}')
    return synced_data, synced_data_last_update_timestamp
