import copy
import json
import logging
import os
import sys
import urllib.parse
from pathlib import Path

import requests
from flask import Flask, request, render_template

file = Path(__file__).resolve()
parent, root = file.parent, file.parents[1]
sys.path.append(str(root))

import set_secret
from featureFlow.help_functions import get_components_config, get_issue_dict_separate, \
    get_startreck_client, get_issues_params_template, startrek_url, get_issue_dict_cumulative
from flow_constants import status_checklists, status_issues, dev_issue_params

logger = logging.getLogger('waitress')
logger.setLevel(logging.INFO)

set_secret.set_secrets()
app = Flask(__name__)

startrek_headers = {
    'Authorization': f'OAuth {os.environ["STARTRECK_TOKEN"]}',
    'Content-Type': 'application/json',
    'User-Agent': 'curl/7.53.1',
    'Connection': 'close',
    'Accept': '*/*'
}
last_issues = []


@app.before_request
def log_the_request():
    logger.info(request)
    body = request.json
    if body:
        logger.info(body)


@app.after_request
def after_request(response):
    if not any(mimetype in response.mimetype for mimetype in ["image"]):
        response_body = response.get_data()
        logger.info(f'Response: {response_body}')
    return response


@app.route("/")
def check_of_functioning():
    return render_template(
        "result.html",
        issues=last_issues
    )


@app.route("/manager/newstatus", methods=['POST'])
def new_status():
    data = request.json
    checklist_status = set_checklist(data["key"], data['status'].lower())
    issues_status = add_issues(data["key"], data['status'].lower())
    return {
        'checklists_status': checklist_status,
        'issues_status': issues_status
    }


@app.route("/manager/make_form", methods=['POST'])
def make_form():
    data = request.json
    issue = get_startreck_client().issues[data['key']]
    goal = issue.goals[0] if len(issue.goals) > 0 else ''
    if goal != '':
        goal = '&goal_id=' + goal.id
    text_start = """
    <{Форма для заведения задачи в разработку
{{iframe src=//forms.yandex-team.ru/surveys/98697?iframe=1"""
    summary = '&project_name=' + urllib.parse.quote_plus(issue.summary)
    main_task = '&main_task=' + urllib.parse.quote_plus(issue.key)
    text_end = " frameborder=0 width=100% height=660px scrolling=no}} }>"
    text = text_start + summary + main_task + goal + text_end
    logger.info(text)
    issue.comments.create(text=text)
    issue.update(
        tags={'remove': ['summon_dev_form']},
        ignore_version_change=True
    )
    return "ok"


@app.route("/dev/issue", methods=['POST'])
def create_dev_issue():
    answers = {}
    for key in request.form.keys():
        question_resp = json.loads(request.form[key])
        answers[question_resp['question']['slug']] = question_resp['value']
    logger.info(f'answers: {answers}')
    issue_params = get_issues_params_template(request.args, answers)
    issue_queue = issue_params['queue']

    components_config = get_components_config(issue_queue, answers)
    issues = []

    if dev_issue_params[issue_queue].get('task_for_each_component', True) & (len(components_config.keys()) > 0):
        for component in components_config:
            current_issue_params = get_issue_dict_separate(copy.deepcopy(issue_params), component, components_config[component])
            logger.info(current_issue_params)
            issue = get_startreck_client().issues.create(**current_issue_params)
            issues.append(issue)
    else:
        issue_params = get_issue_dict_cumulative(issue_params, components_config)
        logger.info(issue_params)
        issue = get_startreck_client().issues.create(**issue_params)
        issues.append(issue)
    global last_issues
    last_issues = (issues + last_issues)[:10]
    keys = ""
    for issue in issues:
        keys += issue.key
    return keys


def set_checklist(key, status):
    queue = key[:key.find('-')]
    if queue not in status_checklists:
        return "no checklists for queue found"
    if status not in status_checklists[queue].keys():
        return "no changes for status needed"
    checklist_for_status = status_checklists[queue][status]
    checklist_now = requests.get(
        f'{startrek_url}/v2/issues/{key}/checklistItems',
        headers=startrek_headers
    ).json()
    checklist_to_add = []
    for new_checklist_item in checklist_for_status:
        is_item_already_present = False
        for old_checklist_item in checklist_now:
            if old_checklist_item['text'] == new_checklist_item:
                is_item_already_present = True
        if not is_item_already_present:
            checklist_to_add.append({
                'text': new_checklist_item
            })
    r = requests.post(
        f'{startrek_url}/v2/issues/{key}/checklistItems',
        data=json.dumps(checklist_to_add),
        headers=startrek_headers
    )
    return "ok"


def add_issues(key, status):
    queue = key[:key.find('-')]
    if queue not in status_issues:
        return "no tickets for queue found"
    if status not in status_issues[queue].keys():
        return "no changes for status needed"
    main_issue = get_startreck_client().issues[key]
    issues_to_create = []
    for issue in status_issues[queue][status]:
        is_item_already_present = False
        for link in main_issue.links:
            if issue['summary'] == link.object.display:
                is_item_already_present = True
        if not is_item_already_present:
            issues_to_create.append(issue)
    for issue in issues_to_create:
        get_startreck_client().issues.create(
            queue=queue,
            summary=issue['summary'],
            parent=key,
            type={'name': issue.get('type', 'task')},
            description=issue.get('description', ''),
        )
    return "ok"
