# -*- coding: utf-8 -*-
import logging
import os
import sys
import traceback
from datetime import datetime
from time import sleep, time
from retrying import retry

import urllib3
import yaml
from startrek_client import Startrek

urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)
log = logging.getLogger(__name__)
log.setLevel(logging.INFO)
formatter = logging.Formatter(fmt="%(asctime)s %(levelname)s: %(message)s", datefmt="%H:%M:%S")
ch = logging.StreamHandler(sys.stdout)
ch.setLevel(logging.INFO)
ch.setFormatter(formatter)
log.addHandler(ch)

log_st = logging.getLogger("startrek_client.connection")
log_st.setLevel(logging.ERROR)
log_st.addHandler(ch)

format = '%Y-%m-%dT%H:%M:%S.%f'
scriptPath = os.path.dirname(os.path.abspath(__file__) + '/SupportQueueMonitoring')
yaconfig = yaml.load(open(os.path.dirname(scriptPath) + "/config.yaml"))
allowed_link_type = yaconfig["link_type"]
client = Startrek(useragent="curl/7.53.1", token=yaconfig["myToken"])


def find_correct_transition(ticket, to_status):
    all_transitions = []
    transitions = client.issues[ticket.key].transitions.get_all()
    for transition in transitions:
        if transition.to.key == to_status:
            log.info('	found transition %s' % transition.id)
            return transition.id
        all_transitions.append(transition.id)
    log.error('	Cant find transition, find only %s' % all_transitions)
    return None


@retry(stop_max_attempt_number=3, wait_fixed=100)
def count_opened_and_closed(links):
    opened = 0
    closed = 0
    for link in links:
        if link.status.key == 'closed':
            closed += 1
        else:
            opened += 1
    return opened, closed


@retry(stop_max_attempt_number=3, wait_fixed=100)
def is_not_duplicate(issue):
    if issue.status.key == 'closed':
        if issue.resolution.key == 'duplicate':
            return False
    return True


@retry(stop_max_attempt_number=3, wait_fixed=100)
def get_last_status_change(issue):
    status_changes_list = list(issue.changelog.get_all(field='status'))
    if len(status_changes_list) > 0:
        last_status_change_datetime = list(issue.changelog.get_all(field='status'))[-1]["updatedAt"]
    else:
        last_status_change_datetime = "1800-10-19T09:55:23.583+0000"
    return datetime.strptime(last_status_change_datetime[:-5], format)


@retry(stop_max_attempt_number=3, wait_fixed=100)
def change_status_and_comment(checking_issue, changing_issue, rule_change_dict):
    checking_issue_status = checking_issue.status.key
    changing_issue_status = changing_issue.status.key
    changed = False
    checking_issue_last_changed = get_last_status_change(checking_issue)
    changing_issue_last_changed = get_last_status_change(changing_issue)
    is_task_chaged_earlier = changing_issue_last_changed < checking_issue_last_changed

    # проверяем. есть ли текущий статус в списке на изменение
    if checking_issue_status in rule_change_dict.keys() and is_task_chaged_earlier and is_not_duplicate(checking_issue):
        log.info('	status in list of changes: %s' % checking_issue_status)
        has_current_status_param = "current_status" in rule_change_dict[checking_issue_status]
        # проверяем, не находится ли тикет уже в нужном статусе
        if changing_issue_status != rule_change_dict[checking_issue_status]["status_needed"]:
            log.info('	ticket is not in status needed: %s' % checking_issue_status)
            # проверяем, есть ли в словаре условие по статусу изменяемого тикета
            if not has_current_status_param:
                log.info(
                    '	has no condition for current status, make transition to %s' %
                    rule_change_dict[checking_issue_status]["status_needed"]
                )
                transition = find_correct_transition(
                    changing_issue,
                    rule_change_dict[checking_issue_status]["status_needed"]
                )
                if transition:
                    changing_issue.transitions[transition].execute()
                    changed = True
            else:
                log.info('	has condition for current status')
                dict_curr_status_rule = rule_change_dict[checking_issue_status]["current_status"]
                if changing_issue_status == dict_curr_status_rule:
                    log.info(
                        '	condition for current status satisfies, make transition to %s' %
                        rule_change_dict[checking_issue_status]["status_needed"]
                    )
                    transition = find_correct_transition(
                        changing_issue,
                        rule_change_dict[checking_issue_status]["status_needed"]
                    )
                    if transition:
                        changing_issue.transitions[transition].execute()
                        changed = True
                else:
                    log.info('	condition for current status NOT satisfies')
            if changed:
                log.info("	changed ticket %s from %s to %s" % (
                    changing_issue.key, changing_issue_status,
                    rule_change_dict[checking_issue_status]["status_needed"]))
            # если нужен комментарий, то пишем или копируем его
            if changed and rule_change_dict[checking_issue_status]["need_comment"]:
                if "need_close_comment" in rule_change_dict[checking_issue_status]:
                    opened_closed = count_opened_and_closed(linked_tickets)
                    message = "Еще одна связанная задача завершена. Всего открытых тикетов - %s, закрытых тикетов - " \
                              "%s." % opened_closed
                    changing_issue.comments.create(text=message)
                    log.info('	create comment: %s' % message)
                else:
                    comments = list(checking_issue.comments.get_all())
                    last_comment = comments[len(comments) - 1].text
                    changing_issue.comments.create(text=last_comment)
                    log.info('	create comment: %s' % last_comment)
        else:
            log.info('	ticket ia not in status needed: %s' % checking_issue_status)
    else:
        if is_task_chaged_earlier:
            log.info('	checking issue changed earlier than changing issue')
        elif not is_not_duplicate(checking_issue):
            log.info('	ticket has resolution "duplicate", skip')
        else:
            log.info('	status NOT in list of changes: %s' % checking_issue_status)


start_time = time()
issues = client.issues.find('Queue: DISKSUP AND Resolution: empty()')
status_map_to_supp = yaconfig["status_map_to"]
status_map_from_supp = yaconfig["status_map_from"]
for issue in issues:
    log.info(' ')
    try:
        log.info('analyzing ' + issue.key)
        links = client.issues[issue.key].links.get_all()
        log.info('	links founded: %s' % len(links))
        linked_tickets = [link for link in links if (link.type.id in allowed_link_type) & (
                link.object.key[:link.object.key.find('-')] != 'DISKSUP')]
        log.info('	linked tickets only: %s' % len(linked_tickets))
        for link in links:
            if (link.type.id in allowed_link_type) & (link.object.key[:link.object.key.find('-')] != 'DISKSUP'):
                log.info('	ticket: %s' % link.object.key)
                linked_issue = client.issues[link.object.key]
                current_link_status = linked_issue.status.key
                current_issue_status = issue.status.key
                change_status_and_comment(issue, linked_issue, status_map_from_supp)
                linked_issue = client.issues[link.object.key]
                issue = client.issues[issue.key]
                change_status_and_comment(linked_issue, issue, status_map_to_supp)
    except:
        exc_type, exc_value, exc_traceback = sys.exc_info()
        lines = traceback.format_exception(exc_type, exc_value, exc_traceback)
        if 'permissions denied' in lines:
            lines = '	permission denied to ticket'
        log.error(lines)

e = int(time() - start_time)
log.info("time spent: " + '{:02d}:{:02d}:{:02d}'.format(e // 3600, (e % 3600 // 60), e % 60))
