# -*- coding: utf-8 -*-
import datetime
import json
import os
import re
import smtplib
import sys
from collections import defaultdict
from email.mime.multipart import MIMEMultipart
from email.mime.text import MIMEText
from pathlib import Path
import yt.wrapper as yt

import pytz
import requests
import urllib3
from retrying import retry

urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)

sys.path.append(str(Path(__file__).parent.parent.parent))
from set_secret import set_secret

set_secret.set_secrets()

local_tz = pytz.timezone('Europe/Moscow')
ab_host = 'https://ab.yandex-team.ru/'
ab_headers = {'Authorization': 'OAuth ' + os.environ['AB_TOKEN']}
fromaddr = 'robot-pinkie-pie@yandex-team.ru'
fromname = 'Pinkie Pie'
mypass = os.environ['PWD']
services = {
    '56': {'name': 'Веб Почта', 'toaddr': ['mail-experiments@yandex-team.ru']},
    '71': {'name': 'Тач Почта', 'toaddr': ['mail-experiments@yandex-team.ru']},
    '95': {'name': 'Моб Почта', 'toaddr': ['mail-experiments@yandex-team.ru']},
    '85': {'name': 'Диск', 'toaddr': ['disk-experiments@yandex-team.ru']},
    '162': {'name': '360', 'toaddr': []},
}
table_exps = "//home/mailfront/qa/exp-config"
yt.config["proxy"]["url"] = "hahn"
yt.config["token"] = os.environ["YT_TOKEN"]


def compare_versions(configid):
    # get current online config version
    r = requests.get(
        f'{ab_host}api/config/{config}/history?limit=1&action=TAG&tag=online',
        headers=ab_headers,
        verify=False
    ).json()
    current_version = r[0]['version']
    # get last checked config version from file
    prev_version = read_from_stat(configid)
    if not prev_version:
        prev_version = int(current_version) - 1

    if int(current_version) != int(prev_version):
        print('current version: %s' % current_version)
        print('previous version: %s' % prev_version)
        # get diff page. there is no such api method
        diff_page = requests.get(
            f'{ab_host}config/{configid}/diff/{prev_version}/{current_version}',
            headers=ab_headers,
            verify=False
        ).text
        # find testids and percentage change in html
        data = re.findall("config_data = ([^\n]*)", diff_page)[0]
        data = json.loads(data)["node_root"]["children"]

        diff = defaultdict(dict)
        for i in range(0, len(data)):
            for exp in data[i]['children']:
                for testid in exp['children']:
                    r = requests.get(
                        f'{ab_host}api/task/{testid["ticket"]}',
                        headers=ab_headers,
                        verify=False
                    ).json()
                    if 'testids' not in diff[r['title']]:
                        diff[r['title']]['testids'] = defaultdict(dict)
                    diff[r['title']]['testids'][testid['testid']] = {'percent': testid['diff'][0]['percent']['data']}
                    diff[r['title']]['testids'][testid['testid']]['ticket'] = testid['ticket']
                    diff[r['title']]['testids'][testid['testid']]['title'] = testid['title']
                    if 'services' in exp['restrictions'][0]:
                        diff[r['title']]['services'] = exp['restrictions'][0]['services']
        print(diff)

        analyze_diff(configid, diff)
        # update last checked version in report
        send_to_stat(configid, current_version)


def analyze_diff(configid, diff):
    exp_percents = {}
    for exp_name in diff:
        text = ''
        additinal_text = ''
        maillists = []
        if configid != '162':
            maillists = services[configid]['toaddr']
        elif 'mail' in diff[exp_name]['services']:
            maillists.append('mail-experiments@yandex-team.ru')
        elif 'disk' in diff[exp_name]['services']:
            maillists.append('disk-experiments@yandex-team.ru')
        testids = diff[exp_name]['testids']
        for testid in testids:
            title = testids[testid]['title']
            ticket = testids[testid]['ticket']
            if testids[testid]['percent'][0] is None and testids[testid]['percent'][1] is None:
                break
            if testids[testid]['percent'][0] is None:
                text = f'Запущен на {testids[testid]["percent"][1]}% эксперимент {exp_name}<br>' \
                       f'<br><a href=https://st.yandex-team.ru/{ticket}>{ticket}</a><br>'
                additinal_text += f' - [{testid}] <a href=https://ab.yandex-team.ru/testid/{testid}>{title}</a><br>'
                exp_percents.update({int(testid): [int(testids[testid]["percent"][1]), ticket, title]})
            elif testids[testid]['percent'][1] is None:
                text = f'<a href=https://st.yandex-team.ru/{ticket}>Эксперимент</a> убран из конфига'
                exp_percents.update({int(testid): [0, ticket, title]})
            elif float(testids[testid]['percent'][0]) < float(testids[testid]['percent'][1]):
                text = f'Процент <a href=https://st.yandex-team.ru/{ticket}>эксперимента</a> увеличен до ' \
                       f'{testids[testid]["percent"][1]}% '
                exp_percents.update({int(testid): [int(testids[testid]["percent"][1]), ticket, title]})
                break
            elif float(testids[testid]['percent'][0]) > float(testids[testid]['percent'][1]):
                text = f'Процент <a href=https://st.yandex-team.ru/{ticket}>эксперимента</a> уменьшен до ' \
                       f'{testids[testid]["percent"][1]}% '
                exp_percents.update({int(testid): [int(testids[testid]["percent"][1]), ticket, title]})
                break
        if text != '':
            send_message(configid, exp_name, text + additinal_text, maillists)
    if exp_percents:
        send_data_to_yt(exp_percents)


@retry(stop_max_attempt_number=3, wait_fixed=10000)
def send_message(configid, name, body, maillists):
    msg = MIMEMultipart()
    msg['From'] = f'{fromname} <{fromaddr}>'
    msg['To'] = ', '.join(maillists)
    msg['Subject'] = "[" + services[configid]['name'] + "] " + name

    msg.attach(MIMEText(body, 'html', 'utf-8'))

    server = smtplib.SMTP_SSL('smtp.yandex-team.ru', 465, timeout=10)
    server.set_debuglevel(1)
    server.login(fromaddr, mypass)
    text = msg.as_string()
    server.sendmail(fromaddr, maillists, text)
    server.quit()


@retry(stop_max_attempt_number=3, wait_fixed=3000)
def send_to_stat(configid, version):
    data = [
        {
            "fielddate": datetime.datetime.now(local_tz).strftime(str("%Y-%m-%d %H:%M:%S")),
            "config": configid,
            "version": version
        }
    ]
    r = requests.post(
        'https://upload.stat.yandex-team.ru/_api/report/data',
        headers={'Authorization': 'OAuth ' + os.environ["STAT_TOKEN"]},
        data={
            'name': 'Mail/Testing/ExpConfigVersions',
            'scale': 's',
            'data': json.dumps({'values': data}),
        },
    )
    print('Stat response %s' % r.status_code)
    attempt_number = 1
    while r.status_code != 200 and attempt_number < 5:
        r = requests.post(
            'https://upload.stat.yandex-team.ru/_api/report/data',
            headers={'Authorization': 'OAuth ' + os.environ["STAT_TOKEN"]},
            data={
                'name': 'Mail/Testing/ExpConfigVersions',
                'scale': 's',
                'data': json.dumps({'values': data}),
            },
        )
        print('Stat response %s' % r.status_code)
        attempt_number += 1
    # print (r.text)


@retry(stop_max_attempt_number=3, wait_fixed=3000)
def read_from_stat(configid):
    min_date = datetime.date.today() - datetime.timedelta(days=45)
    tomorrow = datetime.date.today() + datetime.timedelta(days=1)
    r = requests.get(
        f'https://stat.yandex-team.ru/_api/statreport/json/Mail/Testing/ExpConfigVersions',
        data={
            'config': configid,
            'date_min': min_date,
            'date_max': tomorrow,
            'sort_field': datetime
        },
        headers={'Authorization': 'OAuth ' + os.environ["STAT_TOKEN"]},
    )
    if r.json()['values'] == []:
        return 0
    else:
        return r.json()['values'][0]['version']


@retry(stop_max_attempt_number=3, wait_fixed=3000)
def get_table_data():
    table_data = {}
    for i in yt.read_table(table_exps, format=yt.JsonFormat(attributes={"encode_utf8": False})):
        table_data.update({i.get("exp_id"): [i.get("percent"), i.get("ticket"), i.get("title")]})
    return table_data


@retry(stop_max_attempt_number=3, wait_fixed=10000)
def send_data_to_yt(new_data):
    old_data = get_table_data()
    merged = old_data.copy()
    merged.update(new_data)
    merged_data = merged
    values_list = []
    for i in merged_data:
        values_list.append({
            "exp_id": i,
            "percent": merged_data.get(i)[0],
            "ticket": merged_data.get(i)[1],
            "title": merged_data.get(i)[2]
        })
    yt.write_table(table_exps, values_list, format=yt.JsonFormat(attributes={"encode_utf8": False}))


if __name__ == '__main__':
    for config in services.keys():
        compare_versions(config)
