# -*- coding: utf-8 -*-
import datetime
import json
import logging
import os
import time

import requests
import urllib3
import yaml
from datetime import datetime, timedelta, date
from retrying import retry

urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)
logging.basicConfig(
    format='%(filename)s[LINE:%(lineno)d]# %(levelname)-8s [%(asctime)s]  %(message)s', level=logging.INFO)
scriptPath = os.path.dirname(os.path.abspath(__file__) + '/SupportQueueMonitoring')
yaconfig = yaml.load(open(os.path.dirname(scriptPath) + '/config.yaml'))
month = date.today() - timedelta(days=(date.today().day - 1))
dateLimit = int(time.mktime(month.timetuple())) * 1000
robots = yaconfig['robots']


def is_assessor_version(version):
    if 'asses' in version.lower() or 'ases' in version.lower() or 'acces' in version.lower() \
            or 'асес'.decode('utf-8') in version.lower():
        return True


def is_monitoring_version(version):
    if 'мониторинг'.decode('utf-8') in version.lower() or 'monitoring' in version.lower() \
            or 'регулярный'.decode('utf-8') in version.lower():
        return True


def use_for_autotests_computing(version):
    if independentAutotestVersion:
        if 'autotest' in version.lower():
            return True
        else:
            return False
    return not is_assessor_version(version) and not is_monitoring_version(version)


def created_by_robot(testrun):
    return testrun['createdBy'] in robots


def use_for_autotest(testrun):
    return testrun['launcherInfo']['external'] and created_by_robot(testrun)


def compute_finish_date(version_overview):
    finished_time = version_overview['version']["finishedTime"] / 1000
    if finished_time == 0:
        last_run = max(version_overview['testruns'], key=lambda run: run['finishedTime'])
        finished_time = int(last_run['finishedTime']) / 1000
    version_date = date.fromtimestamp(finished_time)
    return str(version_date)


def cluster_runs(testruns):
    sorted_runs = sorted(testruns[:], key=lambda r: r['finishedTime'])
    cluster_set = []
    current_cluster = [sorted_runs[0]]
    for index in range(1, len(sorted_runs)):
        run = sorted_runs[index]
        if run['startedTime'] - current_cluster[len(current_cluster) - 1]['finishedTime'] < 120000:  # 3 минута
            current_cluster.append(run)
        else:
            cluster_set.append(current_cluster)
            current_cluster = [run]
    cluster_set.append(current_cluster)
    return cluster_set


def cluster_time(testrun):
    started_time = datetime.fromtimestamp(int(testrun['finishedTime']) / 1000)
    return started_time - timedelta(hours=started_time.time().hour % 12)


def cluster_date(testrun):
    started_time = datetime.fromtimestamp(int(testrun['finishedTime']) / 1000)
    return started_time - timedelta(hours=started_time.time().hour)


def testrun_cluster_key(testrun):
    """
    Создаем ключ вида <testsuiteid | имя рана> - <год-месяц-день (00|12):00:00> что бы отличать раны
    разбираемые до обеда и после
    :param testrun: объект рана из версии
    :return: ключ по которому будет идентифицировать ран.
    """
    if 'testSuite' in testrun:
        run_title = testrun['testSuite']['id']
    else:
        run_title = testrun['title']

    return run_title + ' - ' + cluster_time(testrun).strftime('%Y-%m-%d %H:00:00')


def is_nightly_run(testrun):
    return cluster_time(testrun).hour == 0


def cluster_runs_by_day_and_night(testruns):
    """
    Get test run clusters. Only latest test run on the night for specific test suite or test run name
    must be in cluster
    :param testruns: list of runs for specific version
    :return: run_map
    """
    run_map = {}
    for run in testruns:
        if is_last_quarter(run['startedTime']) and use_for_autotest(testrun=run):
            cluster_key = testrun_cluster_key(run)
            if not run_map.get(cluster_key):
                run_map[cluster_key] = run
                continue
            else:
                run_time = int(run['finishedTime']) / 1000
                current_saved_time = int(run_map[cluster_key]['finishedTime']) / 1000
                if run_time > current_saved_time:
                    run_map[cluster_key] = run
                continue
    return run_map


def cluster_by_date(testruns):
    """
    Return only daily runs clustered by date
    :param testruns: list of runs for specific version
    :return: run_map: Map<date, [testrun]>
    """
    run_map = {}
    for run in testruns:
        if is_last_quarter(run['startedTime']) and use_for_autotest(testrun=run):
            cluster_key = run['finishedTime']
            if not run_map.get(cluster_key):
                run_map[cluster_key] = []
            run_map[cluster_key].append(run)
    return run_map


def clusterize_by_suite(testruns):
    run_map = {}
    for run in testruns:
        if is_last_quarter(run['startedTime']):
            cluster_key = testrun_cluster_key(run)
            if not run_map.get(cluster_key):
                run_map[cluster_key] = [run]
                continue
            else:
                run_map[cluster_key].append(run)
                continue
    return run_map


# from 01.08.2018
def is_last_quarter(created_date):
    if created_date >= dateLimit:
        return True
    return False


def is_not_empty(testruns):
    if not testruns:
        return False
    return True


def is_finished(single_run):
    if single_run['status'] == 'FINISHED':
        return True
    return False


@retry(stop_max_attempt_number=3, wait_fixed=100)
def autotest_data():
    url = 'https://testpalm.yandex-team.ru:443/api/version/' + parentId
    req_headers = {'TestPalm-Api-Token': yaconfig['AUTH_TP'], 'Content-Type': 'application/json'}
    all_versions = requests.get(url, headers=req_headers).json()
    versions = list()
    autotest = list()

    for single_version in all_versions:
        if is_last_quarter(single_version['startedTime'])\
                and use_for_autotests_computing(single_version['title']):
            versions.append(single_version['title'])
        else:
            continue

    for autotest_version in versions:
        url = 'https://testpalm.yandex-team.ru:443/api/version/' + parentId + '/overview/' + autotest_version
        req_headers = {'TestPalm-Api-Token': yaconfig['AUTH_TP'], 'Content-Type': 'application/json'}
        response = requests.get(url, headers=req_headers)
        version_overview = response.json()
        version = dict()
        ver_title = version_overview['version']['title']

        autotest_runs = list(filter(lambda r: use_for_autotest(r), version_overview['testruns']))

        if len(autotest_runs) == 0:
            logging.info('Skip empty version %s' % ver_title)
            continue

        if cluster_by_suite:
            test_run_clusters = clusterize_by_suite(autotest_runs)
            summary_execution_time = 0
            for cluster in test_run_clusters.values():
                cluster_execution_time = 0
                for run in cluster:
                    run_execution_time = run['duration']
                    run_start_time = min(cluster, key=lambda r: r['startedTime'])['startedTime']
                    run_end_time = max(cluster, key=lambda r: r['finishedTime'])['finishedTime']
                    exec_time = min(run_end_time - run_start_time, run_execution_time)
                    cluster_execution_time += exec_time
                cluster_average_time = cluster_execution_time / len(cluster)
                summary_execution_time += cluster_average_time
            version_run_time = summary_execution_time / 1000 / 60  # minutes
        else:
            test_run_clusters = cluster_runs(autotest_runs)
            summary_execution_time = 0
            for cluster in test_run_clusters:
                cluster_execution_time = 0
                for run in cluster:
                    cluster_execution_time += run['executionTime']
                run_start_time = min(cluster, key=lambda r: r['startedTime'])['startedTime']
                cluster_end_time = max(cluster, key=lambda r: r['finishedTime'])['finishedTime']
                exec_time = min(cluster_end_time - run_start_time, cluster_execution_time)

                summary_execution_time += min(exec_time, cluster_execution_time)
            average_execution_time = summary_execution_time/1000/60 / len(test_run_clusters)
            version_run_time = average_execution_time  # minutes

        if version_run_time == 0:
            logging.info('Skip empty version %s' % ver_title)
            continue
        else:
            logging.info('Computed autotest time for %s: %s ' % (ver_title, str(version_run_time)))
            version[ver_title] = version_run_time

        fin_date = compute_finish_date(version_overview)
        version = [fin_date, version_run_time, ver_title]
        autotest.append(version)

    for version in autotest:
        data = [
            {'queue': parentId,
             'fielddate': version[0],
             'duration': version[1],
             'version': version[2]}
        ]

        logging.info(' Data is ready %s' % data)

        r = requests.post(
            'https://upload.stat.yandex-team.ru/_api/report/data',
            headers={'Authorization': 'OAuth %s' % yaconfig['AUTH_STAT']},
            data={
                'name': 'Mail/Others/AutotestDetailed',
                'scale': 'd',
                'data': json.dumps({'values': data})
            }
        )

        logging.info(' Stat response: %s' % r.text)


all_queues = yaconfig['parentId']
for queue_data in all_queues:
    for parentId in queue_data:
        queue_params = queue_data[parentId]
        independentAutotestVersion = queue_params['independentAutotestVersion']
        cluster_by_suite = queue_params['clusterBySuite']
        autotest_data()
