#include "counters_worker.h"

#include <mail/ratesrv/src/router/router.h>

#include "json/get_callbacks.h"
#include "json/increase_callbacks.h"

#include "json/request_parser.h"
#include "json/response_maker.h"

#include <mail/ratesrv/src/common/format.h>
#include <mail/ratesrv/src/common/types.h>

#include <yplatform/yield.h>
#include <yplatform/find.h>

#include <string>

namespace NRateSrv::NHandlers {

using namespace yplatform::time_traits;

TCountersWorker::TCountersWorker(THttpStreamPtr httpStream, ERequestMode mode)
    : Ctx(boost::make_shared<TContext>(httpStream->ctx()->uniq_id()))
    , HttpStream(std::move(httpStream))
    , Mode(mode)
    , StartedAt(clock::now())
{}

void TCountersWorker::operator()(TYieldCtx yieldCtx, EStatus status) {
    if (status != EStatus::Ok) {
        return Complete(status);
    }

    reenter (yieldCtx) {
        yield ParseRequest(yieldCtx);
        yield SendToRouter(yieldCtx);

        Complete(EStatus::Ok);
    }
}

void TCountersWorker::ParseRequest(TCallback callback) {
    switch (Mode) {
        case ERequestMode::Get:
            Callbacks = std::make_unique<NJson::TGetCallbacks>();
            break;
        case ERequestMode::Increase:
            Callbacks = std::make_unique<NJson::TIncreaseCallbacks>();
            break;
    }

    EStatus status = EStatus::Ok;
    if (!NJson::ParseRequest(HttpStream->request()->raw_body, Callbacks.get())) {
        status = EStatus::FailToParse;
    }

    callback(status);
}

void TCountersWorker::SendToRouter(TCallback callback) {
    auto router = yplatform::find<NRouter::TRouter>("router");
    auto routerCallback = [callback = std::move(callback), &response = Response](TCounterResponse routerResponse) {
        response = std::move(routerResponse);
        callback(EStatus::Ok);
    };

    switch (Mode) {
        case ERequestMode::Get:
            return router->AsyncGet(Ctx, Callbacks->GetRequest(), std::move(routerCallback));
        case ERequestMode::Increase:
            return router->AsyncIncrease(Ctx, Callbacks->GetRequest(), std::move(routerCallback));
    }
}

void TCountersWorker::Complete(EStatus status) {
    auto code = ymod_webserver::codes::ok;
    bool isJson = true;
    std::string response;

    if (status == EStatus::Ok) {
        response = NJson::MakeCounterValuesResponse(Response);
    } else {
        code = ymod_webserver::codes::bad_request;
        isJson = false;
        auto [offset, description] = Callbacks->GetLastError();
        response = Format("%1% [offset: %2%]", description, offset);
    }

    HttpStream->set_code(code);
    if (isJson) {
        HttpStream->set_content_type("application", "json");
    }
    HttpStream->result_body(response);
}

} // namespace NRateSrv::NHandlers
