#pragma once

#include "handlers/http/counters_get.h"
#include "handlers/http/counters_increase.h"
#include "handlers/http/ping.h"
#include "handlers/http/handler.h"

#include "tvm_check.h"

#include <ymod_webserver/server.h>
#include <ymod_messenger/module.h>

#include <yplatform/find.h>
#include <yplatform/ptree.h>
#include <yplatform/reactor.h>
#include <yplatform/log.h>

#include <exception>
#include <memory>
#include <unordered_set>

namespace NRateSrv {

class TRateSrvImpl : public yplatform::log::contains_logger {
public:
    explicit TRateSrvImpl(yplatform::reactor& reactor);

    void Init(const yplatform::ptree& configuration);
    void Stop();

private:
    void ReadConfiguration(const yplatform::ptree& configuration);

    void BindHttpHandlers();
    void BindHttpHandler(std::shared_ptr<NHandlers::NHttp::IHandler> handler);

    bool CheckAccess(NHandlers::NHttp::THttpStreamPtr stream);

private:
    yplatform::reactor& Reactor;

    bool TvmEnable;
    std::unordered_set<std::string> AllowedTvmClients;
};

TRateSrvImpl::TRateSrvImpl(yplatform::reactor& reactor)
    : Reactor(reactor)
{}

void TRateSrvImpl::Init(const yplatform::ptree& configuration) {
    ReadConfiguration(configuration);
    BindHttpHandlers();
}

void TRateSrvImpl::ReadConfiguration(const yplatform::ptree& configuration) {
    auto tvmPt = configuration.get_child("tvm");
    TvmEnable = tvmPt.get<bool>("enable");
    if (TvmEnable) {
        auto range = tvmPt.equal_range("clients");
        for (auto it = range.first; it != range.second; ++it) {
            AllowedTvmClients.insert(it->second.data());
        }
    }
}

void TRateSrvImpl::Stop() {
    auto messenger = yplatform::find<ymod_messenger::module>("messenger");
    messenger->stop();

    auto webServer = yplatform::find<ymod_webserver::server>("web_server");
    webServer->stop();
}

void TRateSrvImpl::BindHttpHandlers() {
    BindHttpHandler(std::make_shared<NHandlers::NHttp::TPing>());
    BindHttpHandler(std::make_shared<NHandlers::NHttp::TCountersGet>(Reactor));
    BindHttpHandler(std::make_shared<NHandlers::NHttp::TCountersIncrease>(Reactor));
}

void TRateSrvImpl::BindHttpHandler(std::shared_ptr<NHandlers::NHttp::IHandler> handler) {
    auto httpCallback = [this, handler, tvmEnable = TvmEnable](NHandlers::NHttp::THttpStreamPtr stream) {
        if (stream->request()->method != handler->Method()) {
            stream->result(ymod_webserver::codes::method_not_allowed);
            return;
        }

        if (tvmEnable && handler->NeedTvm() && !CheckAccess(stream)) {
            return;
        }

        try {
            handler->Execute(stream);
        } catch (const std::exception& exp) {
            YLOG_CTX_LOCAL(stream->ctx(), fatal) << "Unhandled std exception in http handler: " << exp.what();
            stream->result(ymod_webserver::codes::internal_server_error, "Unhandled std exception");
        } catch (...) {
            YLOG_CTX_LOCAL(stream->ctx(), fatal) << "Unhandled unknown exception in http handler";
            stream->result(ymod_webserver::codes::internal_server_error, "Unhandled unknown exception");
        }
    };
    auto webServer = yplatform::find<ymod_webserver::server>("web_server");
    webServer->bind("", handler->Paths(), std::move(httpCallback));
}

bool TRateSrvImpl::CheckAccess(NHandlers::NHttp::THttpStreamPtr stream) {
    auto [success, client] = CheckTvm(stream->request());
    if (!success) {
        stream->result(ymod_webserver::codes::internal_server_error);
        return false;
    }

    if (client.empty()) {
        stream->result(ymod_webserver::codes::unauthorized);
        return false;
    }

    stream->ctx()->custom_log_data.emplace("tvm_client_id", client);

    if (AllowedTvmClients.count(client) == 0) {
        stream->result(ymod_webserver::codes::unauthorized, "Unknown TVM application");
        return false;
    }

    return true;
}

} // namespace NRateSrv
