#include "counters_worker.h"
#include "node_client.h"
#include "storage_client.h"
#include "util.h"

#include <mail/ratesrv/src/logger.h>
#include <mail/ratesrv/src/common/post_wrapper.h>

#include <yplatform/yield.h>
#include <yplatform/find.h>

#include <spdlog/details/format.h>

#include <util/string/cast.h>

#include <boost/asio/post.hpp>

namespace NRateSrv::NRouter {

TCountersWorker::TCountersWorker(
    yplatform::reactor& reactor,
    TConfigurationPtr configuration,
    TNetworkAgentPtr networkAgent,
    TNodeManagerPtr nodeManager,
    TTaskContextPtr ctx,
    TCounterRequest request,
    ERequestMode mode,
    TCallback callback
)
    : Reactor(reactor)
    , Strand(*Reactor.io())
    , Configuration(std::move(configuration))
    , NetworkAgent(std::move(networkAgent))
    , NodeManager(std::move(nodeManager))
    , Ctx(std::move(ctx))
    , Request(std::move(request))
    , RequestSize(Request.size())
    , Mode(mode)
    , Callback(std::move(callback))
    , HasLocalPart(false)
    , FailsCount(0)
    , ReadyParts(0)
{}

void TCountersWorker::operator()(TYieldCtx yieldCtx) {
    reenter (yieldCtx) {
        yield SplitRequest(yieldCtx);

        yield boost::asio::post(Strand, [yieldCtx, this] {
            SendToDestination(std::move(yieldCtx));
        });

        Complete();
    }
}

void TCountersWorker::SplitRequest(TYieldCtx yieldCtx) {
    size_t index = 0;
    for (auto& [id, counter] : Request) {
        TSplitName splitName;

        if (!SplitName(counter.Name, splitName)) {
            Response[id].State = ECounterState::InvalidName;
            ++StatesCount[ECounterState::InvalidName];
            continue;
        }

        auto bucket = CalcBucketNumber(counter.Name, Configuration->GetBucketCount(), Configuration->GetHashSeed());
        size_t nodeNum = 0;

        if (NodeManager->Count() > 1) {
            nodeNum = CalcNodeNumber(
                bucket,
                NodeManager->Count(),
                Configuration->GetNodeHashSalt(),
                Configuration->GetHashSeed());
        }

        Ids.push_back(std::move(id));
        Parts[nodeNum].Add(
            splitName.Group,
            splitName.Limit,
            std::move(splitName.Domain),
            std::move(splitName.Key),
            index++,
            counter.Value,
            bucket);
    }

    yieldCtx();
}

void TCountersWorker::SendToDestination(TYieldCtx yieldCtx) {
    if (Parts.empty()) {
        return yieldCtx();
    }

    for (auto& [nodeNum, request] : Parts) {
        const auto& node = NodeManager->Get(nodeNum);
        if (node.IsLocal()) {
            SendToLocalStorage(std::move(request), 0, yieldCtx);
        } else {
            SendToRemoteNode(nodeNum, std::move(request), yieldCtx);
        }
    }
}

void TCountersWorker::SendToRemoteNode(size_t nodeNum, NStorage::TRequest request, TYieldCtx yieldCtx) {
    auto handler = [yieldCtx = std::move(yieldCtx), this]
        (size_t attempts, NStorage::TResponse response, NStorage::TRequest request)
    {
        auto failsCount = attempts > 0 ? attempts - 1 : 0;
        if (response.empty()) {
            SendToLocalStorage(std::move(request), failsCount, std::move(yieldCtx));
        } else {
            HandleValues(failsCount, std::move(response), std::move(yieldCtx));
        }
    };
    auto nodeClient = std::make_shared<TNodeClient>(
        Reactor,
        NetworkAgent,
        NodeManager,
        Configuration,
        Ctx,
        std::move(request),
        Mode,
        nodeNum,
        MakePostWrapper(Strand, std::move(handler)));
    yplatform::spawn(std::move(nodeClient));
}

void TCountersWorker::SendToLocalStorage(NStorage::TRequest request, size_t failsCount, TYieldCtx yieldCtx) {
    HasLocalPart = true;
    auto handler = [yieldCtx = std::move(yieldCtx), failsCount, this](NStorage::TResponse response) {
        HandleValues(failsCount, std::move(response), std::move(yieldCtx));
    };
    auto storageClient = std::make_shared<TStorageClient>(
        std::move(request),
        Mode,
        MakePostWrapper(Strand, std::move(handler)));
    yplatform::spawn(Reactor.io()->get_executor(), std::move(storageClient));
}

void TCountersWorker::HandleValues(size_t failsCount, NStorage::TResponse response, TYieldCtx yieldCtx) {
    FailsCount += failsCount;
    StorageResponse.merge(std::move(response));
    if (++ReadyParts == Parts.size()) {
        yieldCtx();
    }
}

void TCountersWorker::Complete() {
    for (auto& [index, value] : StorageResponse) {
        ++StatesCount[value.State];
        Response.emplace(std::move(Ids[index]), std::move(value));
    }

    WriteLog();
    Callback(std::move(Response));
}

void TCountersWorker::WriteLog() {
    auto prefix = static_cast<std::string>(ToString(Mode));

    NJson::TJsonValue states;
    for (const auto& [state, count] : StatesCount) {
        states[ToString(state)] = count;
        auto cnt = count;
        if (state == ECounterState::Ok) {
            RATESRV_LOG_COUNTER(Ctx, notice, prefix + "_counter_ok_states", cnt);
        } else if (state == ECounterState::Exceeded && Mode == ERequestMode::Increase) {
            RATESRV_LOG_COUNTER(Ctx, notice, prefix + "_counter_exceeded_states", cnt);
        } else {
            RATESRV_LOG_COUNTER(Ctx, error, prefix + "_counter_error_states", cnt);
        }
    }

    RATESRV_LOG_COUNTER(Ctx, error, "node_fails_count", FailsCount)
}

} // namespace NRateSrv::NRouter
