#include "node.h"

namespace NRateSrv::NRouter {

TNode::TNode(std::string address, bool local)
    : Address(std::move(address))
    , Local(local)
    , BanNumber(0)
    , BanVotes(0)
    , ResetBanVotes(0)
    , ResetBanAttempts(0)
{}

const std::string& TNode::GetAddress() const {
    return Address;
}

bool TNode::IsLocal() const {
    return Local;
}

bool TNode::IsBanned() const {
    auto guard = Guard(Lock);
    return BanNumber > 0;
}

bool TNode::AllowTryToResetBan() {
    auto guard = Guard(Lock);
    if (BannedUntil >= TClock::now() || ResetBanAttempts == 0) {
        return false;
    }
    --ResetBanAttempts;
    return true;
}

TNode::TDuration TNode::Ban(TDuration baseDuration, size_t quorum, bool force) {
    auto now = TClock::now();
    auto duration = TDuration::zero();
    auto guard = Guard(Lock);

    if (LastBanAttempt + baseDuration < now) {
        BanVotes = 0;
    }

    LastBanAttempt = now;
    if (now <= BannedUntil || (!force && (quorum == 0 || ++BanVotes < quorum))) {
        return duration;
    }

    BanVotes = 0;
    LastBanAttempt = TTimePoint::min();

    ResetBanVotes = quorum;
    ResetBanAttempts = quorum;

    duration = CalcBanDuration(baseDuration, BanNumber++);
    BannedUntil = now + duration;

    return duration;
}

int TNode::ResetBan(bool force) {
    auto guard = Guard(Lock);
    if (!force && (ResetBanVotes == 0 || --ResetBanVotes > 0)) {
        return 0;
    }

    ResetBanVotes = 0;
    ResetBanAttempts = 0;
    int banNumber = BanNumber;
    BanNumber = 0;
    BanVotes = 0;
    BannedUntil = TTimePoint::min();
    LastBanAttempt = TTimePoint::min();

    return banNumber;
}

TNode::TDuration TNode::CalcBanDuration(TDuration base, int num) {
    return base * (num + 1);
}

} // namespace NRateSrv::NRouter
