#include "node_client.h"
#include "network_protocol.h"

#include <mail/ratesrv/src/common/post_wrapper.h>

#include <yplatform/yield.h>

#include <boost/asio/post.hpp>

namespace NRateSrv::NRouter {

TNodeClient::TNodeClient(
    yplatform::reactor& reactor,
    TNetworkAgentPtr networkAgent,
    TNodeManagerPtr nodeManager,
    TConfigurationPtr configuration,
    TTaskContextPtr ctx,
    NStorage::TRequest request,
    ERequestMode mode,
    size_t nodeNum,
    TCallback callback
)
    : Strand(*reactor.io())
    , Timer(Strand.context())
    , NetworkAgent(std::move(networkAgent))
    , NodeManager(std::move(nodeManager))
    , Configuration(std::move(configuration))
    , Ctx(std::move(ctx))
    , Request(std::move(request))
    , Mode(mode)
    , NodeNum(nodeNum)
    , Callback(std::move(callback))
    , Attempts(0)
    , CurrentRequestId(0)
{}

void TNodeClient::operator()(TYieldCtx yieldCtx, ui64 requestId, NStorage::TResponse response, bool isTimeout) {
    if (requestId != CurrentRequestId) {
        return;
    }

    reenter (yieldCtx) {
        for (;;) {
            if (++Attempts > Configuration->GetMaxAttempts()) {
                yield break;
            }

            if (!SetDestination()) {
                yield break;
            }

            yield boost::asio::post(Strand, [yieldCtx, this]() {
                SendRequest(std::move(yieldCtx));
            });

            if (CurrentRequestId == 0) {
                continue;
            }

            if (!isTimeout) {
                Timer.cancel();
                yield break;
            }

            HandleTimeout();
        }
    }

    if (yieldCtx.is_complete()) {
        CurrentRequestId = 0;
        if (response.empty()) {
            Callback(Attempts, {}, std::move(Request));
        } else {
            Callback(Attempts, std::move(response), {});
        }
    }
}

bool TNodeClient::SetDestination() {
    auto [success, destination] = NodeManager->GetActing(NodeNum, UsedNodes);
    if (success) {
        Destination = destination;
        const auto& node = NodeManager->Get(Destination);
        success = !node.IsLocal();
    }
    return success;
}

void TNodeClient::SendRequest(TYieldCtx yieldCtx) {
    auto callback = MakePostWrapper(Strand, yieldCtx);
    CurrentRequestId = NetworkAgent->SendRequest(
        NodeManager,
        Ctx,
        Destination,
        Request,
        Mode == ERequestMode::Get ? EMessageTypes::GetRequest : EMessageTypes::IncreaseRequest,
        callback);

    if (CurrentRequestId == 0) {
        UsedNodes.insert(Destination);
        callback(0, NStorage::TResponse{}, false);
    } else {
        SetTimer(std::move(yieldCtx));
    }
}

void TNodeClient::SetTimer(TYieldCtx yieldCtx) {
    auto handler = [callback = MakePostWrapper(Strand, std::move(yieldCtx)), requestId = CurrentRequestId]
        (const boost::system::error_code& ec)
    {
        if (ec == boost::asio::error::operation_aborted) {
            return;
        }
        callback(requestId, NStorage::TResponse{}, true);
    };

    Timer.expires_after(Configuration->GetTimeout());
    Timer.async_wait(std::move(handler));
}

void TNodeClient::HandleTimeout() {
    NetworkAgent->CompleteRequest(CurrentRequestId, false);
    CurrentRequestId = 0;

    if (Mode == ERequestMode::Increase ||
        ++NodeAttempts[Destination] >= Configuration->GetMaxAttemptsForNode())
    {
        UsedNodes.insert(Destination);
    }
}

} // namespace NRateSrv::NRouter
