#include "node_manager.h"

#include <numeric>

namespace NRateSrv::NRouter {

TNodeManager::TNodeManager(TNodes nodes)
    : Nodes(std::move(nodes))
    , AlternateNodes(Nodes.size())
    , AlternateCounts(Nodes.size(), 0)
{
    std::iota(AlternateNodes.begin(), AlternateNodes.end(), 0);
}

TNode& TNodeManager::Get(size_t nodeNum) {
    return Nodes.at(nodeNum);
}

const TNode& TNodeManager::Get(size_t nodeNum) const {
    return Nodes.at(nodeNum);
}

size_t TNodeManager::Count() const {
    return Nodes.size();
}

std::pair<bool, size_t> TNodeManager::GetActing(size_t nodeNum, const TUsedNumNodes& usedNumNodes) {
    bool success = true;
    auto nextNode = nodeNum;

    auto guard = Guard(AlternateLock);

    for (;;) {
        auto& node = Nodes.at(nextNode);
        if (node.IsLocal()) {
            break;
        }
        const auto used = usedNumNodes.count(nextNode) > 0;

        auto& alternate = AlternateNodes[nextNode];
        if (!used) {
            if (!node.IsBanned()) {
                if (alternate != nextNode) {
                    --AlternateCounts[alternate];
                    alternate = nextNode;
                }
                break;
            }
            if (node.AllowTryToResetBan()) {
                break;
            }
        }

        if (alternate != nextNode) {
            nextNode = alternate;
            continue;
        }

        auto candidate = nextNode;
        auto neighbor = nextNode;
        auto minCount = Nodes.size();

        do {
            ++neighbor %= Nodes.size();
            if (AlternateCounts[neighbor] < minCount &&
                AlternateNodes[neighbor] == neighbor &&
                usedNumNodes.count(neighbor) == 0)
            {
                minCount = AlternateCounts[neighbor];
                candidate = neighbor;
            }
        } while (neighbor != nextNode);

        if (candidate == nextNode) {
            success = false;
            break;
        }

        if (!used) {
            AlternateNodes[nextNode] = candidate;
            ++AlternateCounts[candidate];
        }
        nextNode = candidate;
    }

    return {success, nextNode};
}

} // namespace NRateSrv::NRouter
