#include "counter_updater.h"

#include <algorithm>
#include <chrono>
#include <limits>

namespace NRateSrv::NStorage {

TCounterUpdater::TCounterUpdater(
    TCounterValue& value,
    const TLimitConf& configuration,
    ui64 increaseValue,
    size_t bucket
)
    : Value(value)
    , Configuration(configuration)
    , IncreaseValue(increaseValue)
    , Bucket(bucket)
{}

TCounter TCounterUpdater::operator()(const TCounter* counter) {
    Value.State = ECounterState::Ok;
    auto updatedCounter = counter ? Update(counter) : Create();

    if (IncreaseValue > 0) {
        if (updatedCounter.Value + IncreaseValue > Configuration.Threshold && !Configuration.IgnoreThreshold) {
            Value.State = ECounterState::Exceeded;
        } else {
            updatedCounter.Value += IncreaseValue;
        }
    }

    Value.Current = updatedCounter.Value;
    Value.Available = CalcAvailable(updatedCounter.Value, Configuration.Threshold);

    return updatedCounter;
}

TCounter TCounterUpdater::Update(const TCounter* counter) {
    auto updatedCounter = *counter;
    auto diff = std::chrono::duration_cast<TInterval>(TClock::now() - counter->LastUpdate);
    ui64 recoveryCount = diff / Configuration.RecoveryInterval;

    if (recoveryCount > 0) {
        updatedCounter.Value -= std::min(
            counter->Value,
            recoveryCount * Configuration.RecoveryRate
        );
        updatedCounter.LastUpdate += Configuration.RecoveryInterval * recoveryCount;
    }

    return updatedCounter;
}

TCounter TCounterUpdater::Create() {
    TCounter counter;
    counter.Bucket = Bucket;
    counter.LastUpdate = TClock::now();
    return counter;
}

i64 TCounterUpdater::CalcAvailable(ui64 current, ui64 limit) {
    i64 available = 0;
    constexpr ui64 intMax = std::numeric_limits<i64>::max();

    if (current > limit) {
        available = -static_cast<i64>(std::min(current - limit, intMax));
    } else if (limit > current) {
        available = static_cast<i64>(std::min(limit - current, intMax));
    }

    return available;
}

} // namespace NRateSrv::NStorage
