#include "database.h"
#include "counter_updater.h"

namespace NRateSrv::NStorage {

void TDatabase::Init(TGroups&& groups) {
    Groups = std::move(groups);
}

TResponse TDatabase::Get(TRequest request) {
    return ProcessGroups(request.GetGroups(), false);
}

TResponse TDatabase::Increase(TRequest request) {
    return ProcessGroups(request.GetGroups(), true);
}

size_t TDatabase::Size() {
    size_t size = 0;
    for (auto& [groupName, group] : Groups) {
        for (auto& [limitName, limit] : group.Limits) {
            size += limit.Repository->Size();
        }
    }
    return size;
}

TResponse TDatabase::ProcessGroups(TRequest::TGroups groups, bool isIncrease) {
    TResponse response;

    for (auto& [groupName, limits] : groups) {
        TGroup* group = nullptr;
        auto groupIt = Groups.find(groupName);
        if (groupIt != Groups.end()) {
            group = &groupIt->second;
        }

        response.merge(ProcessLimits(group, std::move(limits), isIncrease));
    }

    return response;
}

TResponse TDatabase::ProcessLimits(TGroup* group, TRequest::TLimits limits, bool isIncrease) {
    TResponse response;

    for (auto& [limitName, keys] : limits) {
        TLimit* limit = nullptr;
        if (group) {
            auto limitIt = group->Limits.find(limitName);
            if (limitIt != group->Limits.end()) {
                limit = &limitIt->second;
            }
        }

        response.merge(ProcessKeys(group, limit, std::move(keys), isIncrease));
    }

    return response;
}

TResponse TDatabase::ProcessKeys(TGroup* group, TLimit* limit, TRequest::TKeys keys, bool isIncrease) {
    TResponse response;
    TRepository::TUpdateHandles handles;

    for (auto& key : keys) {
        auto& value = response[key.Index];
        if (!group) {
            value.State = ECounterState::GroupNotFound;
        } else if (!limit) {
            value.State = ECounterState::LimitNotFound;
        } else {
            if (key.Domain.empty()) {
                key.Domain = "default";
            }
            auto confIt = limit->Configuration.find(key.Domain);
            if (confIt == limit->Configuration.end()) {
                if (key.Domain != "default") {
                    confIt = limit->Configuration.find("default");
                }
                if (confIt == limit->Configuration.end()) {
                    value.State = ECounterState::Unknown;
                    continue;
                }
            }

            TRepository::TUpdateHandle handle;
            handle.Key = std::move(key.Key);
            handle.Callback = TCounterUpdater(value, confIt->second, isIncrease ? key.Value : 0, key.Bucket);
            handles.push_back(std::move(handle));
        }
    }

    if (limit && !handles.empty()) {
        limit->Repository->Update(std::move(handles), isIncrease);
    }

    return response;
}

} // namespace NRateSrv::NStorage
