#pragma once

#include "common.h"

#include <util/system/spinlock.h>
#include <util/system/types.h>

#include <atomic>
#include <functional>
#include <memory>
#include <shared_mutex>
#include <string>
#include <unordered_map>
#include <utility>
#include <vector>

namespace NRateSrv::NStorage {

class TRepository {
public:
    using TReadCallback = std::function<bool(const std::string& key, const TCounter& counter)>;
    using TUpdateCallback = std::function<TCounter(const TCounter* counter)>;
    using TEraseCallback = std::function<bool(const TCounter& counter)>;

    struct TUpdateHandle {
        std::string Key;
        TUpdateCallback Callback;
    };

    using TUpdateHandles = std::vector<TUpdateHandle>;
    using TEraseKeys = std::vector<std::string>;

    explicit TRepository(size_t partCount);

    void Read(TReadCallback callback);
    void Update(TUpdateHandles handles, bool createNonExistent);
    void Erase(TEraseKeys keys, TEraseCallback callback);
    size_t Size();

private:
    using TCounterPair = std::pair<TCounter, TAdaptiveLock>;

    struct TPart {
        std::unordered_map<std::string, TCounterPair> Values;
        std::shared_mutex Lock;
        std::atomic_ullong Size = 0;
    };

    using TParts = std::vector<TPart>;

    enum class EUpdatePolicy {
        ReadWithSkipNonExistent,
        Read,
        Write,
    };

    using TSeparatedUpdateHandles = std::unordered_map<size_t, TUpdateHandles>;
    using TSeparatedEraseKeys = std::unordered_map<size_t, TEraseKeys>;

    size_t CalcPartNumber(const std::string& key);
    void UpdateParts(TSeparatedUpdateHandles separatedHandles, EUpdatePolicy policy);
    void UpdatePart(TPart& part, TUpdateHandles handles, EUpdatePolicy policy);
    void EraseFromParts(TSeparatedEraseKeys keys, TEraseCallback callback);

private:
    TParts Parts;
};

using TRepositoryPtr = std::shared_ptr<TRepository>;

} // namespace NRateSrv::NStorage
