#pragma once

#include "database.h"
#include "gc_task.h"
#include "repository.h"
#include "protocol.h"
#include "common.h"
#include "database_size_task.h"

#include <mail/ratesrv/src/scheduler/scheduler.h>
#include <mail/ratesrv/src/common/format.h>

#include <yplatform/find.h>
#include <yplatform/ptree.h>
#include <yplatform/log.h>

#include <yaml-cpp/yaml.h>

#include <memory>
#include <vector>

namespace NRateSrv::NStorage {

class TStorageImpl : public yplatform::log::contains_logger {
public:
    TStorageImpl();

    void Init(const yplatform::ptree& configuration);
    void Stop();

    TResponse Get(TRequest request);
    TResponse Increase(TRequest request);

private:
    void InitDatabase(TDatabase::TGroups groups);
    void InitGarbageCollector(TInterval checkInterval, TInterval ttl, TRepositoryPtr repository);
    void InitDbSizeTask();

    void ReadConfiguration(const yplatform::ptree& configuration, TDatabase::TGroups& groups);
    void ReadGroupConfiguration(const std::string& name, YAML::Node groupNode, TDatabase::TGroups& groups);
    void ReadLimitConfiguration(const std::string& name, YAML::Node limitNode, TDatabase::TGroups::iterator groupIt);
    void ReadDomainConfiguration(const std::string& name, YAML::Node domainNode, TDatabase::TLimit& limit);

private:
    TDatabasePtr Database;
    std::vector<ui64> SchedulerGroupIds;
};

TStorageImpl::TStorageImpl()
    : Database(std::make_shared<TDatabase>())
{}

void TStorageImpl::Init(const yplatform::ptree& configuration) {
    TDatabase::TGroups groups;
    ReadConfiguration(configuration, groups);
    InitDatabase(std::move(groups));
    InitDbSizeTask();
}

void TStorageImpl::InitDatabase(TDatabase::TGroups groups) {
    Database->Init(std::move(groups));
}

void TStorageImpl::InitDbSizeTask() {
    NScheduler::TGroupSettings groupSettings;
    groupSettings.Duration = std::chrono::seconds{5};
    groupSettings.Policy = NScheduler::EExecutionPolicyWhenTaskAdding::WaitInLine;

    auto scheduler = yplatform::find<NScheduler::TScheduler>("scheduler");
    auto groupId = scheduler->CreateGroup(std::move(groupSettings));
    auto taskId = scheduler->AddTask(std::make_unique<TDatabaseSizeTask>(Database), groupId);
    SchedulerGroupIds.push_back(groupId);

    YLOG_L(info) << Format("Add db size task to scheduler, group id %1%, task id %2%", groupId, taskId);
}

void TStorageImpl::InitGarbageCollector(TInterval checkInterval, TInterval ttl, TRepositoryPtr repository) {
    NScheduler::TGroupSettings groupSettings;
    groupSettings.Duration = checkInterval;
    groupSettings.Policy = NScheduler::EExecutionPolicyWhenTaskAdding::WaitInLine;

    auto scheduler = yplatform::find<NScheduler::TScheduler>("scheduler");
    auto groupId = scheduler->CreateGroup(std::move(groupSettings));
    auto taskId = scheduler->AddTask(std::make_unique<TGcTask>(std::move(repository), ttl), groupId);
    SchedulerGroupIds.push_back(groupId);

    YLOG_L(info) << Format("Add repository GC task to scheduler, group id %1%, task id %2%", groupId, taskId);
}


void TStorageImpl::ReadConfiguration(const yplatform::ptree& configuration, TDatabase::TGroups& groups) {
    YAML::Node groupsNode;
    if (auto groupsFilename = configuration.get<std::string>("groups"); !groupsFilename.empty()) {
        groupsNode = YAML::LoadFile(groupsFilename);
    }

    for (auto it = groupsNode.begin(); it != groupsNode.end(); ++it) {
        ReadGroupConfiguration(it->first.as<std::string>(), it->second, groups);
    }
}

void TStorageImpl::ReadGroupConfiguration(const std::string& name, YAML::Node groupNode, TDatabase::TGroups& groups) {
    auto [groupIt, success] = groups.emplace(name, TDatabase::TGroup{});
    if (!success) {
        throw std::runtime_error(Format("Found duplicated group name: %1%", groupIt->first));
    }

    for (auto it = groupNode.begin(); it != groupNode.end(); ++it) {
        ReadLimitConfiguration(it->first.as<std::string>(), it->second, groupIt);
    }
}

void TStorageImpl::ReadLimitConfiguration(
    const std::string& name,
    YAML::Node limitNode,
    TDatabase::TGroups::iterator groupIt)
{
    size_t partCount = limitNode["part_count"].as<size_t>();
    if (partCount == 0) {
        throw std::runtime_error("Part count must be greater than 0");
    }

    auto [limitIt, success] = groupIt->second.Limits.emplace(name, partCount);
    if (!success) {
        throw std::runtime_error(
            Format("Found duplicated limit name %1% in group %2%", limitIt->first, groupIt->first));
    }
    auto& limit = limitIt->second;

    auto gcNode = limitNode["gc"];
    TInterval checkInterval{gcNode["check_interval"].as<TInterval::rep>()};
    TInterval ttl{gcNode["ttl"].as<TInterval::rep>()};
    if (checkInterval <= TInterval::zero() || ttl <= TInterval::zero()) {
        throw std::runtime_error("TTL and check interval must be greater than 0");
    }

    InitGarbageCollector(checkInterval, ttl, limit.Repository);

    auto domainsNode = limitNode["domains"];
    for (auto it = domainsNode.begin(); it != domainsNode.end(); ++it) {
        ReadDomainConfiguration(it->first.as<std::string>(), it->second, limit);
    }

    if (limit.Configuration.count("default") == 0) {
        throw std::runtime_error(
            Format("Default domain of limit %1% in group %2% is undefined", limitIt->first, groupIt->first));
    }
}

void TStorageImpl::ReadDomainConfiguration(const std::string& name, YAML::Node domainNode, TDatabase::TLimit& limit) {
    auto& configuration = limit.Configuration[name];

    configuration.Threshold = domainNode["threshold"].as<ui64>();
    configuration.RecoveryRate = domainNode["recovery_rate"].as<ui64>();
    configuration.IgnoreThreshold = domainNode["ignore_limit"].as<bool>();
    configuration.RecoveryInterval = TInterval{domainNode["recovery_interval"].as<TInterval::rep>()};

    if (configuration.RecoveryInterval <= TInterval::zero()) {
        throw std::runtime_error("Recovery interval must be greater than 0");
    }
}

void TStorageImpl::Stop() {
    auto scheduler = yplatform::find<NScheduler::TScheduler>("scheduler");
    for (ui64 groupId : SchedulerGroupIds) {
        scheduler->RemoveGroup(groupId);
    }
}

TResponse TStorageImpl::Get(TRequest request) {
    return Database->Get(std::move(request));
}

TResponse TStorageImpl::Increase(TRequest request) {
   return Database->Increase(std::move(request));
}

} // namespace NRateSrv::NStorage
