from argparse import ArgumentParser
from collections import namedtuple
from mail.unistat.cpp.cython.meters import (
    AccessLogCountByPath,
    AccessLogRequestTimeHistByPath,
    AccessLogCountByPathAndFirstStatusDigit,
    HttpClientHttpRequestCountByStatus,
    HttpClientHttpRequestTotalTimeHist,
)
from mail.unistat.cpp.cython.logs import (
    AccessTskv,
    HttpClientLog,
)
from py_defs import (
    CounterMeter,
    ServiceLog,
    run
)
import os
import yaml
import json
import logging
import mail.unistat.cpp.cython.logs as logs
import re


logging.basicConfig(level=logging.WARNING, format='[%(asctime)s] [%(levelname)s]: %(message)s')
log = logging.getLogger(__name__)


SERVICES = [
    ['ratesrv', '/ping'],
    ['localhost', '/metadata'],
]


def parse_args():
    parser = ArgumentParser()
    parser.add_argument('-H', '--host', default='::')
    parser.add_argument('-p', '--port', default=8082, type=int)
    parser.add_argument('-d', '--dir', default='.')
    parser.add_argument('-s', action='store_true', help='read file logs from start')
    parser.add_argument('-l', '--log', default='', help='path for yplatform.log')
    parser.add_argument('service_config')
    return parser.parse_args()


def make_service_config(data):
    return ServiceConfig(
        ratesrv_log=os.path.join(os.curdir, data['config']['log']['ratesrv']['sinks'][0]['path']),
        access_log=os.path.join(os.curdir, data['config']['log']['web_server']['sinks'][0]['path']),
        httpclient_log=os.path.join(os.curdir, data['config']['log']['http_client']['sinks'][0]['path']),
    )


ServiceConfig = namedtuple('ServiceConfig', (
    'ratesrv_log',
    'access_log',
    'httpclient_log'
))


def make_http_client_log_meters(services, http_hist_buckets):
    meters = [
        HttpClientHttpRequestTotalTimeHist(http_hist_buckets, handler, service[0] + handler.replace('/', '_') + "_time")
        for service in services
        for handler in service[1:]
    ]
    meters += [
        HttpClientHttpRequestCountByStatus(handler, "count_by_status_" + service[0])
        for service in services
        for handler in service[1:]
    ]

    return meters


def main():
    args = parse_args()
    log.info('chdir %s' % os.path.abspath(args.dir))
    os.chdir(args.dir)

    with open(args.service_config) as f:
        service_config = make_service_config(yaml.load(f, Loader=yaml.FullLoader))

    fast_forward = args.s

    meters = []
    meters.append(CounterMeter("database_size", "ammm"))
    meters.append(CounterMeter("cluster_usage_percent", "avvv"))
    for prefix in ["get", "increase"]:
        for suffix in ["counter_ok_states", "counter_exceeded_states", "counter_error_states", "node_requests"]:
            meters.append(CounterMeter(prefix + "_" + suffix, "summ"))
    main_log = ServiceLog([], meters, fast_forward, service_config.ratesrv_log)

    access_log_metrics = []
    http_hist_buckets = (0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 12, 14, 16, 18, 20, 25, 30, 35, 40, 50, 100, 200, 500, 1000, 2000, 3000)
    for ep in ["/counters/increase", "/counters"]:
        access_log_metrics.append(AccessLogCountByPathAndFirstStatusDigit(ep, "access_log"))
        access_log_metrics.append(AccessLogRequestTimeHistByPath(http_hist_buckets, ep, "access_log_request_time"))
    access_log = AccessTskv([], access_log_metrics, fast_forward, service_config.access_log)

    http_client_log = HttpClientLog([], make_http_client_log_meters(SERVICES, http_hist_buckets), fast_forward, service_config.httpclient_log)

    logs_list = [main_log, access_log, http_client_log]

    run(args.host, args.port, logs_list, args.log, logLevel='info')


if __name__ == '__main__':
    main()

