#include <mail/ratesrv/src/handlers/json/request_parser.h>
#include <mail/ratesrv/src/handlers/json/get_callbacks.h>
#include <mail/ratesrv/src/handlers/json/increase_callbacks.h>

#include <yplatform/zerocopy/streambuf.h>

#include <gtest/gtest.h>

#include <string>

namespace {

using namespace testing;
using namespace NRateSrv::NHandlers::NJson;

class TTestRequestParser : public Test {
protected:
    bool Parse(const std::string& request, TCallbacks* callbacks) {
        yplatform::zerocopy::streambuf buffer(512, 512);
        std::ostream stream(&buffer);

        stream << request << std::flush;

        return ParseRequest(buffer.detach(buffer.end()), callbacks);
    }

    TGetCallbacks MakeGetCallbacks() {
        return TGetCallbacks();
    }

    TIncreaseCallbacks MakeIncreaseCallbacks() {
        return TIncreaseCallbacks();
    }
};

TEST_F(TTestRequestParser, SimpleIncorrect) {
    std::string request = "blabla";
    auto getCallbacks = MakeGetCallbacks();
    auto increaseCallbacks = MakeIncreaseCallbacks();

    EXPECT_FALSE(Parse(request, &getCallbacks));
    EXPECT_FALSE(Parse(request, &increaseCallbacks));

    request = "{}";
    getCallbacks = MakeGetCallbacks();
    increaseCallbacks = MakeIncreaseCallbacks();

    EXPECT_FALSE(Parse(request, &getCallbacks));
    EXPECT_FALSE(Parse(request, &increaseCallbacks));

    request = R"({"counters": "blabla"})";
    getCallbacks = MakeGetCallbacks();
    increaseCallbacks = MakeIncreaseCallbacks();

    EXPECT_FALSE(Parse(request, &getCallbacks));
    EXPECT_FALSE(Parse(request, &increaseCallbacks));
}

TEST_F(TTestRequestParser, EmptyGet) {
    std::string request = R"({"counters": {}})";
    auto callbacks = MakeGetCallbacks();

    ASSERT_TRUE(Parse(request, &callbacks));
    EXPECT_TRUE(callbacks.GetRequest().empty());
}

TEST_F(TTestRequestParser, Get) {
    std::string request = R"({"counters": {"id1": "counter1", "id2": "counter2"}})";
    auto callbacks = MakeGetCallbacks();

    ASSERT_TRUE(Parse(request, &callbacks));
    auto data = callbacks.GetRequest();

    ASSERT_EQ(data.size(), 2u);
    ASSERT_EQ(data.count("id1"), 1u);
    ASSERT_EQ(data.count("id2"), 1u);

    EXPECT_EQ(data["id1"].Name, "counter1");
    EXPECT_EQ(data["id2"].Name, "counter2");
}

TEST_F(TTestRequestParser, GetWithNonUniqueId) {
    std::string request = R"({"counters": {"id1": "counter1", "id1": "counter2"}})";
    auto callbacks = MakeGetCallbacks();

    ASSERT_FALSE(Parse(request, &callbacks));
    EXPECT_EQ(callbacks.GetLastError().second, "Counter id must be unique");
}

TEST_F(TTestRequestParser, GetWithEmptyId) {
    std::string request = R"({"counters": {"id1": "counter1", "": "counter2"}})";
    auto callbacks = MakeGetCallbacks();

    ASSERT_FALSE(Parse(request, &callbacks));
    EXPECT_EQ(callbacks.GetLastError().second, "Counter id must not be empty");
}

TEST_F(TTestRequestParser, GetWithEmptyName) {
    std::string request = R"({"counters": {"id1": "counter1", "id2": ""}})";
    auto callbacks = MakeGetCallbacks();

    ASSERT_FALSE(Parse(request, &callbacks));
    EXPECT_EQ(callbacks.GetLastError().second, "Counter name must not be empty");
}

TEST_F(TTestRequestParser, GetWithIncreaseRequest) {
    std::string request = R"({"counters": {"id1": {"name": "counter1", "value": 2}}})";
    auto callbacks = MakeGetCallbacks();

    ASSERT_FALSE(Parse(request, &callbacks));
}

TEST_F(TTestRequestParser, EmptyIncrease) {
    std::string request = R"({"counters": {}})";
    auto callbacks = MakeIncreaseCallbacks();

    ASSERT_TRUE(Parse(request, &callbacks));
    EXPECT_TRUE(callbacks.GetRequest().empty());
}

TEST_F(TTestRequestParser, Increase) {
    std::string request =
        R"({"counters": {"id1": {"name": "counter1", "value": 100}, "id2": {"name": "counter2", "value": 200}}})";
    auto callbacks = MakeIncreaseCallbacks();

    ASSERT_TRUE(Parse(request, &callbacks));
    auto data = callbacks.GetRequest();

    ASSERT_EQ(data.size(), 2u);
    ASSERT_EQ(data.count("id1"), 1u);
    ASSERT_EQ(data.count("id2"), 1u);

    EXPECT_EQ(data["id1"].Name, "counter1");
    EXPECT_EQ(data["id2"].Name, "counter2");
    EXPECT_EQ(data["id1"].Value, 100u);
    EXPECT_EQ(data["id2"].Value, 200u);
}

TEST_F(TTestRequestParser, IncreaseWithNonUniqueId) {
    std::string request =
        R"({"counters": {"id1": {"name": "counter1", "value": 100}, "id1": {"name": "counter2", "value": 200}}})";
    auto callbacks = MakeIncreaseCallbacks();

    ASSERT_FALSE(Parse(request, &callbacks));
    EXPECT_EQ(callbacks.GetLastError().second, "Counter id must be unique");
}

TEST_F(TTestRequestParser, IncreaseWithEmtpyId) {
    std::string request =
        R"({"counters": {"id1": {"name": "counter1", "value": 100}, "": {"name": "counter2", "value": 200}}})";
    auto callbacks = MakeIncreaseCallbacks();

    ASSERT_FALSE(Parse(request, &callbacks));
    EXPECT_EQ(callbacks.GetLastError().second, "Counter id must not be empty");
}

TEST_F(TTestRequestParser, IncreaseWithEmtpyName) {
    std::string request =
        R"({"counters": {"id1": {"name": "counter1", "value": 100}, "id2": {"name": "", "value": 200}}})";
    auto callbacks = MakeIncreaseCallbacks();

    ASSERT_FALSE(Parse(request, &callbacks));
    EXPECT_EQ(callbacks.GetLastError().second, "Counter name must not be empty");
}

TEST_F(TTestRequestParser, IncreaseWithZeroValue) {
    std::string request =
        R"({"counters": {"id1": {"name": "counter1", "value": 0}, "id2": {"name": "counter2", "value": 200}}})";
    auto callbacks = MakeIncreaseCallbacks();

    ASSERT_FALSE(Parse(request, &callbacks));
    EXPECT_EQ(callbacks.GetLastError().second, "Counter value must be greater than zero");
}

TEST_F(TTestRequestParser, IncreaseWithFloatPointValue) {
    std::string request =
        R"({"counters": {"id1": {"name": "counter1", "value": 100.0}, "id2": {"name": "counter2", "value": 200}}})";
    auto callbacks = MakeIncreaseCallbacks();

    ASSERT_FALSE(Parse(request, &callbacks));
}

TEST_F(TTestRequestParser, IncreaseWithGetRequest) {
    std::string request = R"({"counters": {"id1": "counter1", "id2": "counter2"}})";
    auto callbacks = MakeIncreaseCallbacks();

    ASSERT_FALSE(Parse(request, &callbacks));
}

}
