#include <mail/ratesrv/src/instances/checker/host_checker_impl.h>
#include <mail/ratesrv/ut/mock/http_client.h>

#include <ymod_httpclient/call.h>

#include <gtest/gtest.h>
#include <gmock/gmock.h>

namespace ymod_httpclient {

static inline bool operator ==(const request& lhs, const request& rhs) {
    return lhs.method == rhs.method
        && lhs.url == rhs.url
        && lhs.headers == rhs.headers
        && ((lhs.body && rhs.body && *lhs.body == *rhs.body) || (!lhs.body && !rhs.body));
}

} // namespace ymod_httpclient

namespace {

using namespace testing;
using namespace NRateSrv;
using namespace NRateSrv::NInstances;
using namespace NRateSrv::NMock;

struct TTestHostChecker: Test {
    const yhttp::request HostCheckerRequest = yhttp::request::GET("host/ping");
    const std::shared_ptr<StrictMock<THttpClientMock>> HttpClient = std::make_shared<StrictMock<THttpClientMock>>();
    const THostCheckerPtr Checker = std::make_shared<THostChecker>(HttpClient, Io);
    TContextPtr Context = boost::make_shared<TContext>();

    boost::asio::io_context Io;

    void AsyncCheck(std::string host, NRateSrv::NInstances::IHostChecker::TCallback callback) {
        Checker->AsyncCheck(Context, std::move(host), std::move(callback));
        Io.run();
    }
};

TEST_F(TTestHostChecker, ForHttpErrorShouldReturnFalse) {
    InSequence seq;
    EXPECT_CALL(*HttpClient, async_run(_, HostCheckerRequest, _, _))
        .WillOnce(InvokeArgument<3>(ymod_httpclient::http_error::code::ssl_error, yhttp::response{}));

    AsyncCheck("host", [](auto check) {
        ASSERT_FALSE(check);
    });
}

TEST_F(TTestHostChecker, ForNotOkStatusCodeShouldReturnFalse) {
    InSequence seq;
    EXPECT_CALL(*HttpClient, async_run(_, HostCheckerRequest, _, _))
        .WillOnce(InvokeArgument<3>(boost::system::error_code{}, yhttp::response{ .status = 500 }));

    AsyncCheck("host", [](auto check) {
        ASSERT_FALSE(check);
    });
}

TEST_F(TTestHostChecker, ForGoodRequestShouldReturnTrue) {
    InSequence seq;
    EXPECT_CALL(*HttpClient, async_run(_, HostCheckerRequest, _, _))
        .WillOnce(InvokeArgument<3>(boost::system::error_code{}, yhttp::response{ .status = 200 }));

    AsyncCheck("host", [](auto check) {
        ASSERT_TRUE(check);
    });
}

} // namespace anonymous
