#include <mail/ratesrv/src/instances/qloud/errors.h>
#include <mail/ratesrv/src/instances/qloud/host_getter_impl.h>
#include <mail/ratesrv/ut/mock/http_cluster_client.h>

#include <ymod_httpclient/call.h>

#include <gtest/gtest.h>
#include <gmock/gmock.h>

namespace ymod_httpclient {

static inline bool operator ==(const request& lhs, const request& rhs) {
    return lhs.method == rhs.method
        && lhs.url == rhs.url
        && lhs.headers == rhs.headers
        && ((lhs.body && rhs.body && *lhs.body == *rhs.body) || (!lhs.body && !rhs.body));
}

} // namespace ymod_httpclient

namespace {

using namespace testing;
using namespace NRateSrv;
using namespace NRateSrv::NInstances;
using namespace NRateSrv::NInstances::NQloud;
using namespace NRateSrv::NMock;

struct TTestHostGetter: Test {
    const yhttp::request HostGetterRequest = yhttp::request::GET("/metadata");
    const std::shared_ptr<StrictMock<TClusterClientMock>> HttpClient = std::make_shared<StrictMock<TClusterClientMock>>();
    const THostGetterPtr Getter = std::make_shared<THostGetter>(HttpClient, Io);
    TContextPtr Context = boost::make_shared<TContext>();

    boost::asio::io_context Io;

    void AsyncGet(NRateSrv::NInstances::IHostGetter::TCallback callback) {
        Getter->AsyncGet(Context, std::move(callback));
        Io.run();
    }
};

TEST_F(TTestHostGetter, ForErroOnRequestShouldReturnError) {
    const InSequence sequence;
    EXPECT_CALL(*HttpClient, async_run(_, HostGetterRequest, _, _))
        .WillOnce(InvokeArgument<3>(ymod_httpclient::http_error::code::ssl_error, yhttp::response{}));

    AsyncGet([](auto ec, auto) {
        ASSERT_TRUE(ec);
    });
}

TEST_F(TTestHostGetter, For500StatusCodeShouldReturnError) {
    const InSequence sequence;
    EXPECT_CALL(*HttpClient, async_run(_, HostGetterRequest, _, _))
        .WillOnce(InvokeArgument<3>(boost::system::error_code{}, yhttp::response{ .status = 500 }));

    AsyncGet([](auto ec, auto) {
        ASSERT_TRUE(ec);
        ASSERT_EQ(ec, HttpError);
    });
}

TEST_F(TTestHostGetter, ForBadResponseShouldReturnError) {
    const InSequence sequence;
    const auto jsonResponse = R"({"instances":[})";

    EXPECT_CALL(*HttpClient, async_run(_, HostGetterRequest, _, _))
        .WillOnce(InvokeArgument<3>(boost::system::error_code{}, yhttp::response{ .status = 200, .body = jsonResponse }));

    AsyncGet([](auto ec, auto) {
        ASSERT_TRUE(ec);
        ASSERT_EQ(ec, ParseError);
    });
}

TEST_F(TTestHostGetter, ForGoodResponseShouldReturnHostList) {
    const InSequence sequence;
    const auto jsonResponse = R"({
        "instances": [
        {
          "id": 1,
          "name": "name-1",
          "state": "EXISTING",
          "fqdn": "fqdn-1",
          "ip": "ip-1"
        },
        {
          "id": 2,
          "name": "name-2",
          "state": "NEW",
          "fqdn": "fqdn-2",
          "ip": "ip-2"
        },
        {
          "id": 3,
          "name": "name-3",
          "state": "RESIDING",
          "fqdn": "fqdn-3",
          "ip": "ip-3"
        }
        ]})";

    EXPECT_CALL(*HttpClient, async_run(_, HostGetterRequest, _, _))
        .WillOnce(InvokeArgument<3>(boost::system::error_code{}, yhttp::response{ .status = 200, .body = jsonResponse }));

    AsyncGet([](auto ec, auto response) {
        ASSERT_FALSE(ec);
        EXPECT_THAT(
            response,
            ElementsAre(
                "fqdn-1",
                "fqdn-2",
                "fqdn-3"
            )
        );
    });
}

} // namespace anonymous
