#include <mail/ratesrv/src/instances/hostlist_builder.h>
#include <mail/ratesrv/src/instances/host_checker.h>
#include <mail/ratesrv/src/instances/qloud/errors.h>

#include <mail/ratesrv/ut/mock/host_checker.h>
#include <mail/ratesrv/ut/mock/host_getter.h>

#include <gtest/gtest.h>

namespace {

using namespace testing;
using namespace NRateSrv;
using namespace NRateSrv::NInstances;
using namespace NRateSrv::NInstances::NQloud;
using namespace NRateSrv::NMock;

struct THostListBuilderTest: Test {
    const std::shared_ptr<StrictMock<TMockHostChecker>> Checker = std::make_shared<StrictMock<TMockHostChecker>>();
    const std::shared_ptr<StrictMock<TMockHostGetter>> Getter = std::make_shared<StrictMock<TMockHostGetter>>();
    TContextPtr Context = boost::make_shared<TContext>();
    boost::asio::io_context Io;

    void HostListBuild(THostListBuilder::TCallback callback) {
        yplatform::spawn(
            Io.get_executor(),
            std::make_shared<THostListBuilder>(Context, Getter, Checker, std::move(callback))
        );
        Io.run();
    }
};

TEST_F(THostListBuilderTest, ForErrorOnHostsRequestShouldReturnError) {
    const InSequence sequence;
    EXPECT_CALL(*Getter, AsyncGet(_, _))
        .WillOnce(InvokeArgument<1>(make_error_code(EError::ParseError), THosts{}));

    HostListBuild([](auto ec, auto response) {
        ASSERT_TRUE(ec);
        EXPECT_TRUE(response.LocalHost.empty());
        EXPECT_TRUE(response.Hosts.empty());
        EXPECT_EQ(response.ClusterUsagePersent, 0u);
    });
}

TEST_F(THostListBuilderTest, ForGoodResponseShouldReturnHostList) {
    const InSequence sequence;

    EXPECT_CALL(*Getter, AsyncGet(_, _))
        .WillOnce(InvokeArgument<1>(boost::system::error_code{}, THosts{"fqdn-1", "fqdn-2", "fqdn-3"}));
    EXPECT_CALL(*Checker, AsyncCheck(_,"fqdn-1", _))
        .WillOnce(InvokeArgument<2>(true));
    EXPECT_CALL(*Checker, AsyncCheck(_, "fqdn-2", _))
        .WillOnce(InvokeArgument<2>(false));
    EXPECT_CALL(*Checker, AsyncCheck(_, "fqdn-3", _))
        .WillOnce(InvokeArgument<2>(true));
    EXPECT_CALL(*Getter, LocalHost())
        .WillOnce(Return("localhost"));    

    HostListBuild([](auto ec, auto response) {
        ASSERT_FALSE(ec);
        EXPECT_EQ(response.LocalHost, "localhost");
        EXPECT_THAT(
            response.Hosts,
            ElementsAre(
                "fqdn-1",
                "fqdn-3"
            )
        );
        EXPECT_EQ(response.ClusterUsagePersent, 66u);
    });
}

} // namespace anonymous
