#include <mail/ratesrv/src/storage/counter_updater.h>

#include <gtest/gtest.h>

namespace {

using namespace testing;
using namespace NRateSrv;
using namespace NRateSrv::NStorage;

class TTestCounterUpdater : public Test {
protected:
    const i64 DefaultThreshold = 100;
    const ui64 DefaultRecoveryRate = 10;
    const TInterval DefaultRecoveryInterval = TInterval(10);
    const bool DefaultIgnoreThreshold = true;

    const size_t DefaultIncreaseValue = 10;
    const size_t DefaultBucket = 2;

protected:
    void SetUp() override {
        Configuration.Threshold = DefaultThreshold;
        Configuration.RecoveryRate = DefaultRecoveryRate;
        Configuration.RecoveryInterval = DefaultRecoveryInterval;
        Configuration.IgnoreThreshold = DefaultIgnoreThreshold;
    }

    TLimitConf GetDefaultConfiguration() {
        return Configuration;
    }

private:
    TLimitConf Configuration;
};

TEST_F(TTestCounterUpdater, GetNonExistent) {
    TCounterValue value;
    auto configuration = GetDefaultConfiguration();
    TCounterUpdater updater(value, configuration);
    updater(nullptr);

    EXPECT_EQ(value.State, ECounterState::Ok);
    EXPECT_EQ(value.Current, 0u);
    EXPECT_EQ(value.Available, DefaultThreshold);
}

TEST_F(TTestCounterUpdater, Get) {
    TCounterValue value;
    auto configuration = GetDefaultConfiguration();
    TCounter origCounter;

    origCounter.Value = 14;
    origCounter.Bucket = 15;
    origCounter.LastUpdate = TClock::now();

    TCounterUpdater updater(value, configuration);
    auto counter = updater(&origCounter);

    EXPECT_EQ(value.State, ECounterState::Ok);
    EXPECT_EQ(value.Current, origCounter.Value);
    EXPECT_EQ(value.Available, DefaultThreshold - i64(origCounter.Value));

    EXPECT_EQ(counter.Value, origCounter.Value);
    EXPECT_EQ(counter.Bucket, origCounter.Bucket);
    EXPECT_TRUE(counter.LastUpdate == origCounter.LastUpdate);
}

TEST_F(TTestCounterUpdater, GetExceeded) {
    TCounterValue value;
    auto configuration = GetDefaultConfiguration();
    TCounter origCounter;

    i64 excess = 14;
    origCounter.Value = DefaultThreshold + excess;
    origCounter.Bucket = 15;
    origCounter.LastUpdate = TClock::now();

    TCounterUpdater updater(value, configuration);
    auto counter = updater(&origCounter);

    EXPECT_EQ(value.State, ECounterState::Ok);
    EXPECT_EQ(value.Current, origCounter.Value);
    EXPECT_EQ(value.Available, -excess);

    EXPECT_EQ(counter.Value, origCounter.Value);
    EXPECT_EQ(counter.Bucket, origCounter.Bucket);
    EXPECT_TRUE(counter.LastUpdate == origCounter.LastUpdate);
}

TEST_F(TTestCounterUpdater, GetUpdated) {
    TCounterValue value;
    auto configuration = GetDefaultConfiguration();
    TCounter origCounter;

    origCounter.Value = 90;
    origCounter.Bucket = 15;
    origCounter.LastUpdate = TClock::now() - DefaultRecoveryInterval * 3;

    TCounterUpdater updater(value, configuration);
    auto counter = updater(&origCounter);

    ui64 expectedLoss = 3 * DefaultRecoveryRate;
    EXPECT_EQ(value.State, ECounterState::Ok);
    EXPECT_EQ(value.Current, origCounter.Value - expectedLoss);
    EXPECT_EQ(value.Available, DefaultThreshold - i64(origCounter.Value) + i64(expectedLoss));

    EXPECT_EQ(counter.Value, origCounter.Value - expectedLoss);
    EXPECT_EQ(counter.Bucket, origCounter.Bucket);
    EXPECT_TRUE(counter.LastUpdate > TClock::now() - std::chrono::seconds(1));
}

TEST_F(TTestCounterUpdater, IncreaseNonExistent) {
    TCounterValue value;
    auto configuration = GetDefaultConfiguration();
    TCounterUpdater updater(value, configuration, DefaultIncreaseValue, DefaultBucket);
    auto counter = updater(nullptr);

    EXPECT_EQ(value.State, ECounterState::Ok);
    EXPECT_EQ(value.Current, DefaultIncreaseValue);
    EXPECT_EQ(value.Available, DefaultThreshold - i64(DefaultIncreaseValue));

    EXPECT_EQ(counter.Value, DefaultIncreaseValue);
    EXPECT_EQ(counter.Bucket, DefaultBucket);
    EXPECT_TRUE(counter.LastUpdate > TClock::now() - std::chrono::seconds(1));
}

TEST_F(TTestCounterUpdater, Increase) {
    TCounterValue value;
    auto configuration = GetDefaultConfiguration();
    TCounter origCounter;

    origCounter.Value = 14;
    origCounter.Bucket = 15;
    origCounter.LastUpdate = TClock::now();

    TCounterUpdater updater(value, configuration, DefaultIncreaseValue, DefaultBucket);
    auto counter = updater(&origCounter);

    EXPECT_EQ(value.State, ECounterState::Ok);
    EXPECT_EQ(value.Current, origCounter.Value + DefaultIncreaseValue);
    EXPECT_EQ(value.Available, DefaultThreshold - i64(origCounter.Value) - i64(DefaultIncreaseValue));

    EXPECT_EQ(counter.Value, origCounter.Value + DefaultIncreaseValue);
    EXPECT_EQ(counter.Bucket, origCounter.Bucket);
    EXPECT_TRUE(counter.LastUpdate == origCounter.LastUpdate);
}

TEST_F(TTestCounterUpdater, IncreaseWithSuccessExceeded) {
    TCounterValue value;
    TCounter origCounter;

    origCounter.Value = 10;
    origCounter.Bucket = 15;
    origCounter.LastUpdate = TClock::now();

    auto configuration = GetDefaultConfiguration();
    configuration.IgnoreThreshold = true;

    TCounterUpdater updater(value, configuration, DefaultThreshold, DefaultBucket);
    auto counter = updater(&origCounter);

    EXPECT_EQ(value.State, ECounterState::Ok);
    EXPECT_EQ(value.Current, origCounter.Value + DefaultThreshold);
    EXPECT_EQ(value.Available, -static_cast<i64>(origCounter.Value));

    EXPECT_EQ(counter.Value, origCounter.Value + DefaultThreshold);
    EXPECT_EQ(counter.Bucket, origCounter.Bucket);
    EXPECT_TRUE(counter.LastUpdate == origCounter.LastUpdate);
}

TEST_F(TTestCounterUpdater, IncreaseWithFailedExceeded) {
    TCounterValue value;
    TCounter origCounter;

    origCounter.Value = 10;
    origCounter.Bucket = 15;
    origCounter.LastUpdate = TClock::now();

    auto configuration = GetDefaultConfiguration();
    configuration.IgnoreThreshold = false;

    TCounterUpdater updater(value, configuration, DefaultThreshold, DefaultBucket);
    auto counter = updater(&origCounter);

    EXPECT_EQ(value.State, ECounterState::Exceeded);
    EXPECT_EQ(value.Current, origCounter.Value);
    EXPECT_EQ(value.Available, DefaultThreshold - i64(origCounter.Value));

    EXPECT_EQ(counter.Value, origCounter.Value);
    EXPECT_EQ(counter.Bucket, origCounter.Bucket);
    EXPECT_TRUE(counter.LastUpdate == origCounter.LastUpdate);
}

TEST_F(TTestCounterUpdater, IncreaseUpdated) {
    TCounterValue value;
    auto configuration = GetDefaultConfiguration();
    TCounter origCounter;

    origCounter.Value = 90;
    origCounter.Bucket = 15;
    origCounter.LastUpdate = TClock::now() - DefaultRecoveryInterval * 3;

    TCounterUpdater updater(value, configuration, DefaultIncreaseValue, DefaultBucket);
    auto counter = updater(&origCounter);

    i64 expectedLoss = 3 * DefaultRecoveryRate;
    EXPECT_EQ(value.State, ECounterState::Ok);
    EXPECT_EQ(value.Current, origCounter.Value - expectedLoss + DefaultIncreaseValue);
    EXPECT_EQ(value.Available, DefaultThreshold - i64(origCounter.Value) + expectedLoss - i64(DefaultIncreaseValue));

    EXPECT_EQ(counter.Value, origCounter.Value - expectedLoss + DefaultIncreaseValue);
    EXPECT_EQ(counter.Bucket, origCounter.Bucket);
    EXPECT_TRUE(counter.LastUpdate > TClock::now() - std::chrono::seconds(1));
}

}
