#include <mail/ratesrv/src/storage/database.h>

#include <gtest/gtest.h>

namespace {

using namespace testing;
using namespace NRateSrv;
using namespace NRateSrv::NStorage;

class TTestDatabase : public Test {
protected:
    const size_t DefaultPartCount = 10;

    const i64 DefaultThreshold = 100;
    const ui64 DefaultRecoveryRate = 10u;
    const TInterval DefaultRecoveryInterval = TInterval(10);
    const bool DefaultIgnoreThreshold = false;

protected:
    void SetUp() override {
        TLimitConf configuration;
        configuration.Threshold = DefaultThreshold;
        configuration.RecoveryRate = DefaultRecoveryRate;
        configuration.RecoveryInterval = DefaultRecoveryInterval;
        configuration.IgnoreThreshold = DefaultIgnoreThreshold;

        TLimitConf configuration2 = configuration;
        configuration2.Threshold = 1;

        TDatabase::TLimit limit1(DefaultPartCount);
        TDatabase::TLimit limit2(DefaultPartCount);
        TDatabase::TLimit limit3(DefaultPartCount);

        limit1.Configuration.emplace("default", configuration);
        limit2.Configuration.emplace("default", configuration);
        limit2.Configuration.emplace("alternate", configuration2);

        TDatabase::TGroups groups;
        groups["group1"].Limits.emplace("limit1", std::move(limit1));
        groups["group2"].Limits.emplace("limit2", std::move(limit2));
        groups["group2"].Limits.emplace("limit3", std::move(limit3));
        Database.Init(std::move(groups));
    }

protected:
    TDatabase Database;
};

TEST_F(TTestDatabase, EmptyRequest) {
    EXPECT_TRUE(Database.Get({}).empty());
    EXPECT_TRUE(Database.Increase({}).empty());
    EXPECT_EQ(Database.Size(), 0u);
}

TEST_F(TTestDatabase, NonExistentGroupAndLimit) {
    TRequest request;
    request.Add("unknown_group", "limit", "domain", "key", 4, 10, 1);
    request.Add("group1", "unknown_limit", "domain", "key", 9, 10, 1);
    request.Add("group1", "limit1", "", "key", 2, 10, 1);
    auto response = Database.Get(request);

    ASSERT_EQ(response.size(), 3u);
    ASSERT_EQ(response.count(4), 1u);
    ASSERT_EQ(response.count(9), 1u);
    ASSERT_EQ(response.count(2), 1u);

    EXPECT_EQ(response[4].State, ECounterState::GroupNotFound);
    EXPECT_EQ(response[9].State, ECounterState::LimitNotFound);
    EXPECT_EQ(response[2].State, ECounterState::Ok);
    EXPECT_EQ(Database.Size(), 0u);
}

TEST_F(TTestDatabase, DomainsUsage) {
    {
        TRequest request;
        request.Add("group2", "limit2", "", "key", 0, 10, 1);

        auto response = Database.Increase(request);
        ASSERT_EQ(response.count(0), 1u);
        EXPECT_EQ(response[0].State, ECounterState::Ok);
        EXPECT_EQ(response[0].Current, 10u);
        EXPECT_EQ(response[0].Available, DefaultThreshold - 10);
        EXPECT_EQ(Database.Size(), 1u);
    }

    {
        TRequest request;
        request.Add("group2", "limit2", "nonexisten", "key", 0, 10, 1);

        auto response = Database.Increase(request);
        ASSERT_EQ(response.count(0), 1u);
        EXPECT_EQ(response[0].State, ECounterState::Ok);
        EXPECT_EQ(response[0].Current, 20u);
        EXPECT_EQ(response[0].Available, DefaultThreshold - 20);
        EXPECT_EQ(Database.Size(), 1u);
    }

    {
        TRequest request;
        request.Add("group2", "limit2", "alternate", "key", 0, 10, 1);

        auto response = Database.Increase(request);
        ASSERT_EQ(response.count(0), 1u);
        EXPECT_EQ(response[0].State, ECounterState::Exceeded);
        EXPECT_EQ(response[0].Current, 20u);
        EXPECT_EQ(response[0].Available, -19);
        EXPECT_EQ(Database.Size(), 1u);
    }
}

TEST_F(TTestDatabase, LimitWithoutDefaultConfiguration) {
    TRequest request;
    request.Add("group2", "limit3", "domain", "key", 0, 10, 1);

    {
        auto response = Database.Get(request);
        ASSERT_EQ(response.count(0), 1u);
        EXPECT_EQ(response[0].State, ECounterState::Unknown);
    }

    {
        auto response = Database.Increase(request);
        ASSERT_EQ(response.count(0), 1u);
        EXPECT_EQ(response[0].State, ECounterState::Unknown);
    }
}

}
